<?php

namespace App\Http\Controllers;

use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;
use Lcobucci\JWT\Configuration;
use Lcobucci\JWT\UnencryptedToken;

class AuthController extends Controller
{
    public function apple(Request $request)
    {
        $request->validate([
            'identity_token' => ['required', 'string'],
        ]);

        // Minimal SIWA token parse; robust validation can be added with Apple's JWKS
        $tokenString = $request->string('identity_token');

        try {
            $config = Configuration::forUnsecuredSigner();
            /** @var UnencryptedToken $token */
            $token = $config->parser()->parse($tokenString);
            $claims = $token->claims()->all();
            $sub = $claims['sub'] ?? null;
            if (!$sub) {
                return response()->json(['message' => 'invalid_token'], 422);
            }
        } catch (\Throwable $e) {
            Log::warning('SIWA parse failed', ['error' => $e->getMessage()]);
            return response()->json(['message' => 'invalid_token'], 422);
        }

        $user = User::firstOrCreate(['apple_sub_id' => $sub], [
            'name' => $claims['name'] ?? 'User',
            'email' => $claims['email'] ?? null,
            'password' => Hash::make(Str::random(32)),
        ]);

        $token = $user->createToken('mobile', expiresAt: now()->addMinutes((int) env('SANCTUM_EXPIRATION', 43200)));
        return response()->json([
            'token' => $token->plainTextToken,
            'token_type' => 'Bearer',
            'expires_at' => $token->accessToken->expires_at?->toISOString(),
        ], 201);
    }

    public function refresh(Request $request)
    {
        $user = $request->user();
        $request->user()->currentAccessToken()?->delete();
        $token = $user->createToken('mobile', expiresAt: now()->addMinutes((int) env('SANCTUM_EXPIRATION', 43200)));
        return response()->json([
            'token' => $token->plainTextToken,
            'token_type' => 'Bearer',
            'expires_at' => $token->accessToken->expires_at?->toISOString(),
        ]);
    }
}


