<?php

namespace App\Http\Controllers;

use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;
use Lcobucci\JWT\Configuration;
use Lcobucci\JWT\UnencryptedToken;

class AuthController extends Controller
{
    public function apple(Request $request)
    {
        $request->validate([
            'identity_token' => ['required', 'string'],
        ]);

        // Minimal SIWA token parse; robust validation can be added with Apple's JWKS
        $tokenString = $request->string('identity_token');

        try {
            $parts = explode('.', (string) $tokenString);
            if (count($parts) < 2) {
                return response()->json(['message' => 'invalid_token'], 422);
            }
            $payload = json_decode(self::base64UrlDecode($parts[1]), true) ?: [];
            $sub = $payload['sub'] ?? null;
            if (!$sub) {
                return response()->json(['message' => 'invalid_token'], 422);
            }
        } catch (\Throwable $e) {
            Log::warning('SIWA parse failed', ['error' => $e->getMessage()]);
            return response()->json(['message' => 'invalid_token'], 422);
        }

        $user = User::firstOrCreate(['apple_sub_id' => $sub], [
            'name' => $claims['name'] ?? 'User',
            'email' => $claims['email'] ?? null,
            'password_hash' => Hash::make(Str::random(32)),
        ]);

        $token = $user->createToken('mobile', expiresAt: now()->addMinutes((int) env('SANCTUM_EXPIRATION', 43200)));
        return response()->json([
            'token' => $token->plainTextToken,
            'token_type' => 'Bearer',
            'expires_at' => $token->accessToken->expires_at?->toISOString(),
        ], 201);
    }

    public function refresh(Request $request)
    {
        $user = $request->user();
        $request->user()->currentAccessToken()?->delete();
        $token = $user->createToken('mobile', expiresAt: now()->addMinutes((int) env('SANCTUM_EXPIRATION', 43200)));
        return response()->json([
            'token' => $token->plainTextToken,
            'token_type' => 'Bearer',
            'expires_at' => $token->accessToken->expires_at?->toISOString(),
        ]);
    }

    private static function base64UrlDecode(string $input): string
    {
        $remainder = strlen($input) % 4;
        if ($remainder) {
            $padLen = 4 - $remainder;
            $input .= str_repeat('=', $padLen);
        }
        return base64_decode(strtr($input, '-_', '+/')) ?: '';
    }
}


