<?php

namespace App\Models;

use Carbon\Carbon;
use App\Casts\Upper;
use App\Casts\Boolean;
use App\Models\Samples;
use App\Models\Shipment;

use App\Models\BaseModel;
use App\Helper\Conversions;
use Bkwld\Cloner\Cloneable;
use App\Models\ShipmentLineSizes;
use App\Models\CustomerOrderLines;
use Illuminate\Support\Facades\Cache;
use App\Models\ShipmentLineExftyDates;
use OwenIt\Auditing\Contracts\Auditable;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Database\Eloquent\Casts\Attribute;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Support\Facades\DB;
use App\Services\TotalCacheService;


class ShipmentLine extends BaseModel implements Auditable
{
    use \OwenIt\Auditing\Auditable;
    use SoftDeletes;
    use HasFactory;
    use Cloneable;


    protected $auditEvents = [
        'deleted',
        'updated',
        'saved',
        'restored',
    ];



    public static function boot()
    {
        parent::boot();

        static::deleting(function($model) {
            $model->shipment_line_sizes?->each?->delete();
        });
        static::restoring(function($model) {
            $model->shipment_line_sizes()->withTrashed()->where('deleted_at', '>=', $model->deleted_at)->each(function ($item, $key) { $item->restore(); });
        });

        // Total cache invalidation hooks
        static::saved(function($model) {
            $model->invalidateTotalCache();
        });

        static::updated(function($model) {
            $model->invalidateTotalCache();
        });

        static::deleted(function($model) {
            $model->invalidateTotalCache();
        });

        static::updating(function ($shipmentLine) {
            if ($shipmentLine->isDirty('exfty')) {
                Cache::forget("shipment_line:{$shipmentLine->id}:first_exfty");
            }
        });

        static::updated(function ($shipmentLine) {
            if ($shipmentLine->wasChanged(['collection_date', 'exfty', 'shipment_id'])) {
                $originalShipmentId = (int) ($shipmentLine->getOriginal('shipment_id') ?? 0);
                if ($originalShipmentId) {
                    self::recomputeShipmentTruckFirstCollection($originalShipmentId);
                }
                if ($shipmentLine->shipment_id) {
                    self::recomputeShipmentTruckFirstCollection((int) $shipmentLine->shipment_id);
                }
            }
        });

        static::created(function ($shipmentLine) {
            Cache::forget("shipment_line:{$shipmentLine->id}:first_exfty");
            if ($shipmentLine->shipment_id) {
                self::recomputeShipmentTruckFirstCollection((int) $shipmentLine->shipment_id);
            }
        });

        static::deleted(function ($shipmentLine) {
            if ($shipmentLine->shipment_id) {
                self::recomputeShipmentTruckFirstCollection((int) $shipmentLine->shipment_id);
            }
        });
    }

    /**
     * Invalidate total cache for this shipment line and related entities.
     */
    public function invalidateTotalCache(): void
    {
        try {
            $service = app(TotalCacheService::class);
            
            // Invalidate this shipment line's cache
            $service->invalidateEntity('shipment_line', $this->id);
            
            // Invalidate related customer order line cache
            if ($this->customer_order_lines_id) {
                $service->invalidateEntity('customer_order_line', $this->customer_order_lines_id);
            }
            
            // Invalidate related customer order cache
            if ($this->customer_order_lines && $this->customer_order_lines->customer_orders_id) {
                $service->invalidateEntity('customer_order', $this->customer_order_lines->customer_orders_id);
            }
            
            // Invalidate related shipment cache
            if ($this->shipment_id) {
                $service->invalidateEntity('shipment', $this->shipment_id);
            }
        } catch (\Exception $e) {
            \Log::error("Failed to invalidate total cache for shipment line {$this->id}: " . $e->getMessage());
        }
    }

    protected static function recomputeShipmentTruckFirstCollection(int $shipmentId): void
    {
        $tfc = self::where('shipment_id', $shipmentId)
            ->whereNull('deleted_at')
            ->min(DB::raw('COALESCE(collection_date, exfty)'));
        Shipment::whereKey($shipmentId)->update(['truck_first_collection' => $tfc]);
    }


    protected $casts = [
        'truck_plates'           => Upper::class,
        'sealer_deadline'        => 'date:Y-m-d',
        'exfty'                  => 'date:Y-m-d',
        'collection_date'        => 'date:Y-m-d',
        'delivery_date'          => 'date:Y-m-d',
        'cp_photoshoot_deadline' => 'date:Y-m-d',
        'created_at'             => 'datetime:Y-m-d',
        'updated_at'             => 'datetime:Y-m-d',
        'deleted_at'             => 'datetime:Y-m-d',

        'shipped'                => 'boolean',
        'flag_issue'             => 'boolean',
        'invoiced'               => 'boolean',
        'buttons_required'       => 'boolean',

    ];

	protected $fillable = [
        'shipment_id',
        'customer_order_lines_id',
        'truck_plates',
        'collection_date',
        'delivery_date',
        'complete',
        'rt_invoice',
        'factory_invoice',
        'customs_agency_invoice',
        'created_at',
        'updated_at',
        'cp_notes',
        'cp_photoshoot_deadline',
        'transporter_invoioce',
        'ss_comments',
        'invoiced',
        'buttons',
        'labels',
        'care',
        'pp_sent',
        'buttons_done',
        'labels_done',
        'care_done',
        'no_invoiced',
        'fn_notes',
        'factory_order_yarn',
        'yarn_comments',
        'sealer_deadline',
        'exfty',
        'start_knit_comments',
        'shipped_qty',
        'customer_invoice_no',
        'pl_sent_date',
        'payment_reference',
        'bol_reference',
        'bol_tracking_factory',
        'bol_tracking_rt',
        'container_no',
        'container_eta',
        'asn',
        'asn_transferred_date',
        'asn_transferred_comments',
        'foy_ordered',
        'foy_due',
        'foy_delivered',
        'buttons_required',
        'buttons_due',
        'notes',
        'pre_production_comments',
        'flag_issue',

        'gross_weight',
        'net_weight',
        'no_cartons',

        'season_id',
        'customer_id',
        'factory_id',
        'style_id',
        'design_id',
        'colourway_id',
        'sample_id',
        'price_id',
    ];


    public function scopeSearch($query, $value)
    {
        $sanitizedValue = preg_replace('/[^a-zA-Z0-9\s-]/', '', $value);
        $terms = preg_split('/\s+/', trim($sanitizedValue));

        $query->where(function ($outerQuery) use ($terms) {
            foreach ($terms as $term) {
                $outerQuery->where(function ($q) use ($term) {
                    // Direct columns on shipment_lines
                    $q->orWhere('shipment_lines.rt_invoice', 'like', "%{$term}%")
                      ->orWhere('shipment_lines.id', 'like', "%{$term}%")
                      ->orWhere('shipment_lines.factory_invoice', 'like', "%{$term}%")
                      ->orWhere('shipment_lines.shipment_id', 'like', "%{$term}%");
                      
                    // customer_ref on styles
                    $q->orWhereHas('customer_order_lines.colourways.style_versions.styles', function ($qq) use ($term) {
                        $qq->withoutGlobalScopes(['order'])
                           ->where('styles.customer_ref', 'like', "%{$term}%");
                    });

                    // designs.description via styles -> designs
                    $q->orWhereHas('customer_order_lines.colourways.style_versions.styles', function ($qq) use ($term) {
                        $qq->withoutGlobalScopes(['order'])
                           ->whereHas('designs', function ($qqq) use ($term) {
                               $qqq->withoutGlobalScopes(['order'])
                                   ->where('designs.description', 'like', "%{$term}%");
                           });
                    });

                    // styles.designs_id
                    $q->orWhereHas('customer_order_lines.colourways.style_versions.styles', function ($qq) use ($term) {
                        $qq->withoutGlobalScopes(['order'])
                           ->where('styles.designs_id', 'like', "%{$term}%");
                    });

                    // colourways.name
                    $q->orWhereHas('customer_order_lines.colourways', function ($qq) use ($term) {
                        $qq->withoutGlobalScopes(['order'])
                           ->where('colourways.name', 'like', "%{$term}%");
                    });

                    // style_versions.name
                    $q->orWhereHas('customer_order_lines.colourways.style_versions', function ($qq) use ($term) {
                        $qq->withoutGlobalScopes(['order'])
                           ->where('style_versions.name', 'like', "%{$term}%");
                    });

                    // factories.name via style_versions.factories
                    $q->orWhereHas('customer_order_lines.colourways.style_versions.factories', function ($qq) use ($term) {
                        $qq->withoutGlobalScopes(['order'])
                           ->where('suppliers.name', 'like', "%{$term}%");
                    });

                    // yarn_colours.description and reference
                    $q->orWhereHas('customer_order_lines.colourways.colourway_yarns.yarn_colours', function ($qq) use ($term) {
                        $qq->withoutGlobalScopes(['order'])
                           ->where(function ($qqq) use ($term) {
                               $qqq->where('yarn_colours.description', 'like', "%{$term}%")
                                   ->orWhere('yarn_colours.reference', 'like', "%{$term}%");
                           });
                    });

                    // yarn.description
                    // $q->orWhereHas('customer_order_lines.colourways.colourway_yarns.yarn_colours.yarn', function ($qq) use ($term) {
                    //     $qq->withoutGlobalScopes(['order'])
                    //        ->where('yarn.description', 'like', "%{$term}%");
                    // });

                    // counts.count
                    $q->orWhereHas('customer_order_lines.colourways.colourway_yarns.yarn_colours.yarn.counts', function ($qq) use ($term) {
                        $qq->withoutGlobalScopes(['order'])
                           ->where('counts.count', 'like', "%{$term}%");
                    });

                    // customer_orders.customer_po
                    $q->orWhereHas('customer_order_lines.customer_orders', function ($qq) use ($term) {
                        $qq->withoutGlobalScopes(['order'])
                           ->where('customer_orders.customer_po', 'like', "%{$term}%");
                    });

                    // customers.name via customer_orders.customers
                    $q->orWhereHas('customer_order_lines.customer_orders.customers', function ($qq) use ($term) {
                        $qq->withoutGlobalScopes(['order'])
                           ->where('customers.name', 'like', "%{$term}%");
                    });

                    // shipments.transporter_invoice
                    $q->orWhereHas('shipments', function ($qq) use ($term) {
                        $qq->withoutGlobalScopes(['order'])
                           ->where('shipments.transporter_invoice', 'like', "%{$term}%");
                    });
                });
            }
        });
    }

    public function customer_order_lines()
    {
        return $this->belongsTo(CustomerOrderLines::class);
    }
    public function shipments()
    {
        return $this->belongsTo(Shipment::class, 'shipment_id');
    }
    public function shipment_line_sizes()
    {
        return $this->hasMany(ShipmentLineSizes::class, 'shipment_line_id', 'id');
    }


    // Direct shortcut relationships (cached performance fields)
    public function seasons()
    {
        return $this->belongsTo(Season::class);
    }

    public function customers()
    {
        return $this->belongsTo(Customer::class);
    }

    public function factories()
    {
        return $this->belongsTo(Suppliers::class, 'factory_id');
    }

    public function styles()
    {
        return $this->belongsTo(Style::class);
    }

    public function designs()
    {
        return $this->belongsTo(Design::class);
    }

    public function colourways()
    {
        return $this->belongsTo(Colourway::class);
    }


    public function getTotalPiecesAttribute()
    {
        // Check if the relationship is loaded to avoid lazy-loading and potential hangs
        if ($this->relationLoaded('shipment_line_sizes')) {
            // Use collection sum if already loaded
            return $this->shipment_line_sizes->sum('qty');
        } else {
            // If not loaded, fetch the sum directly from the database to avoid loading all records
            return $this->shipment_line_sizes()->sum('qty');
        }
    }
    public function getTotalPiecesShippedAttribute()
    {
        // Check if the relationship is loaded to avoid lazy-loading and potential hangs
        if ($this->relationLoaded('shipment_line_sizes')) {
            // Use collection sum if already loaded
            return $this->shipment_line_sizes->sum('shipped_qty');
        } else {
            // If not loaded, fetch the sum directly from the database to avoid loading all records
            return $this->shipment_line_sizes()->sum('shipped_qty');
        }
    }


    protected $cachedFirstExfty; // Local variable for caching within the model

    public function getFirstExftyAttribute()
    {
        if (!isset($this->cachedFirstExfty)) {
            $cacheKey = "shipment_line:{$this->id}:first_exfty";

            $this->cachedFirstExfty = Cache::remember($cacheKey, now()->addWeek(), function () {
                return $this->audits()
                    ->whereRaw("JSON_CONTAINS_PATH(old_values, 'one', '$.exfty')")
                    ->oldest()
                    ->first()?->old_values['exfty'] ?? $this->exfty;
            });
        }

        return $this->cachedFirstExfty;
    }

    public function getTotalPricesAttribute(){
        // Try to get cached totals first
        $service = app(TotalCacheService::class);
        $cachedTotals = $service->get('shipment_line', $this->id, 'prices');
        
        if ($cachedTotals) {
            return $cachedTotals;
        }
        
        // Fallback to calculation if no cache
        $quote = 0;
        $quote_base = 0;
        $cmt = 0;
        $subtotal = 0;
        $subtotal_base = 0;
        $transport_budget = 0;
        $transport_currency = null;
        
        foreach($this->shipment_line_sizes as $qty){
            $price = $this->customer_order_lines->customer_order_line_quantities->where('sizes_id', $qty->sizes_id)->first()?->price_model;
            if(!empty($price)){
                $pieces = empty($qty->shipped_qty) ? $qty->qty : $qty->shipped_qty;
                $quote += ($price['quote'] * $pieces);
                $quote_base += ($price['quote_base'] * $pieces);
                $cmt += ($price['cmt'] * $pieces);
                $subtotal += ($price['subtotal'] * $pieces);
                $subtotal_base += ($price['subtotal_base'] * $pieces);
                $transport_budget += ($price['transport_budget_base'] * $pieces);
                $transport_currency = $price['transport_currency'] ?? null;
            }
        }
        
        $totals = [
            'quote' => $quote, 
            'quote_base' => $quote_base, 
            'cmt' => $cmt, 
            'subtotal' => $subtotal, 
            'subtotal_base' => $subtotal_base, 
            'transport_budget' => $transport_budget, 
            'transport_currency' => $transport_currency
        ];
        
        // Cache the calculated totals for next time
        $service->set('shipment_line', $this->id, 'prices', $totals);
        
        return $totals;
	}

    protected $cachedShipmentSample; // Local cache variable

    public function getShipmentSampleAttribute()
    {
        if (!isset($this->cachedShipmentSample)) {
            $cacheKey = "shipment_line:{$this->id}:shipment_sample";
            $tag = "shipment_samples:colourway_{$this->customer_order_lines->colourways_id}";

            $this->cachedShipmentSample = Cache::tags([$tag])->remember($cacheKey, now()->addWeek(), function () {
                $sampleModel = Samples::where('colourways_id', $this->customer_order_lines->colourways_id)
                    ->where('sample_types_id', 7)
                    ->orderBy('created_at', 'desc')
                    ->first();

                if ($sampleModel !== null || $this->shipment_status) {
                    return [
                        'status' => $sampleModel->status ?? $this->shipment_status,
                        'id' => $sampleModel->id ?? null,
                        'comments' => $sampleModel->comments ?? $this->shipment_comments,
                        'date_sent' => $sampleModel->date_sent ?? null,
                        'note' => $sampleModel ? null : '(SS)',
                    ];
                }

                $sampleRequired = json_decode($this->customer_order_lines->customer_orders->customers->samples_required, true)[7] ?? false;
                $carryoverSampleRequired = $this->customer_order_lines->customer_orders->customers->setting('carryover-samples-required')[7] ?? false;

                return [
                    'status' => $this->customer_order_lines->colourways->style_versions->styles->carryover
                        ? ($carryoverSampleRequired ? 'Required' : 'Not Required')
                        : ($sampleRequired ? 'Required' : 'Not Required'),
                    'id' => null,
                    'comments' => null,
                    'date_sent' => null,
                    'note' => $this->customer_order_lines->colourways->style_versions->styles->carryover ? '(Carryover)' : null
                ];
            });
        }

        return $this->cachedShipmentSample;
    }

    protected $cachedSealerSample; // Local cache variable

    public function getSealerSampleAttribute()
    {
        if (!isset($this->cachedSealerSample)) {
            $cacheKey = "shipment_line:{$this->id}:sealer_sample";

            $this->cachedSealerSample = Cache::remember($cacheKey, now()->addWeek(), function () {
                $sampleModel = Samples::where('colourways_id', $this->customer_order_lines->colourways_id)
                    ->where('sample_types_id', 3)
                    ->orderBy('created_at', 'desc')
                    ->first();

                if ($sampleModel !== null || $this->shipment_status) {
                    return [
                        'status' => $sampleModel->status ?? $this->shipment_status,
                        'id' => $sampleModel->id ?? null,
                        'comments' => $sampleModel->comments ?? $this->shipment_comments,
                        'date_sent' => $sampleModel->date_sent ?? null,
                        'note' => $sampleModel ? null : '(SS)',
                    ];
                }

                $sampleRequired = json_decode($this->customer_order_lines->customer_orders->customers->samples_required, true)[3] ?? false;
                $carryoverSampleRequired = $this->customer_order_lines->customer_orders->customers->setting('carryover-samples-required')[3] ?? false;

                return [
                    'status' => $this->customer_order_lines->colourways->style_versions->styles->carryover
                        ? ($carryoverSampleRequired ? 'Required' : 'Not Required')
                        : ($sampleRequired ? 'Required' : 'Not Required'),
                    'id' => null,
                    'comments' => null,
                    'date_sent' => null,
                    'note' => $this->customer_order_lines->colourways->style_versions->styles->carryover ? '(Carryover)' : null
                ];
            });
        }

        return $this->cachedSealerSample;
    }
}
