<?php

namespace App\Models;

use DateTime;
use App\Models\Price;
use App\Casts\Boolean;
use App\Models\Samples;
use App\Casts\TitleCase;
use App\Enums\ColourType;
use App\Helper\Functions;
use App\Models\BaseModel;
use App\Helper\Conversions;
use Illuminate\Support\Str;
use \Bkwld\Cloner\Cloneable;
use App\Models\StyleVersions;
use App\Models\ColourwayYarns;
use App\Models\CustomerOrderLines;
use App\Models\ColourwayAccessories;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Gate;
use Illuminate\Support\Facades\Cache;
use App\Exceptions\DependencyException;
use Illuminate\Database\Eloquent\Scope;
use Illuminate\Support\Facades\Storage;
use OwenIt\Auditing\Contracts\Auditable;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Database\Eloquent\Casts\Attribute;
use Illuminate\Database\Eloquent\Factories\HasFactory;


class Colourways extends BaseModel implements Auditable
{
    use \OwenIt\Auditing\Auditable;
    use SoftDeletes;
    use Cloneable;

    public static function boot()
    {
        parent::boot();

        static::deleting(function($model) {

            // Check for dependent colourways
            if ($model->customer_order_lines()->exists()) {
                throw new DependencyException("Cannot delete: Colourway has dependent customer_order_lines.");
            }

            // Check for dependent colourways
            if ($model->samples()->exists()) {
                throw new DependencyException("Cannot delete: Colourway has dependent samples.");
            }

            $model->colourway_yarns?->each?->delete();
            $model->samples?->each?->delete();
            $model->customer_order_lines?->each?->delete();
            $model->prices?->each?->delete();
            if($model->isForceDeleting()){
                if(File::exists('.'.Storage::url($model->image))){
                    File::delete('.'.Storage::url($model->image));
                }
                if(File::exists('.'.Storage::url($model->img_thumb))){
                    File::delete('.'.Storage::url($model->img_thumb));
                }
            }
        });

        static::restoring(function($model) {
            $model->colourway_yarns()->withTrashed()->where('deleted_at', '>=', $model->deleted_at)->each(function ($item, $key) { $item->restore(); });
            $model->samples()->withTrashed()->where('deleted_at', '>=', $model->deleted_at)->each(function ($item, $key) { $item->restore(); });
            $model->customer_order_lines()->withTrashed()->where('deleted_at', '>=', $model->deleted_at)->each(function ($item, $key) { $item->restore(); });
        });
    }

    protected static function booted()
    {
        static::updated(function ($colourway) {
            ShipmentLine::where('colourway_id', $colourway->id)->update([
                'colourway_id' => $colourway->id,
            ]);
            self::invalidatePriceResolutions($colourway);
        });

        static::saved(function ($colourway) {
            self::invalidatePriceResolutions($colourway);
        });

        static::deleted(function ($colourway) {
            self::invalidatePriceResolutions($colourway);
        });
    }

    // protected static function booted()
    // {
    //     static::addGlobalScope(new ColourwaysUserDepartmentScope);
    // }

    protected $appends = ['thumb_url', 'image_url', 'price_model'];

    protected $cloneable_relations = [
        'colourway_yarns',
    ];

    protected $casts = [
        'name' => TitleCase::class,
        'created_at' => 'datetime:Y-m-d',
        'updated_at' => 'datetime:Y-m-d',
        'deleted_at' => 'datetime:Y-m-d',
        'cancelled' => Boolean::class,
    ];

	protected $fillable = [
        'style_versions_id',
        'name',
        'image',
        'comments',
        'created_at',
        'updated_at',
        'cancelled',
        'accessories',
        'colour_type',
        'img_thumb',
        'deleted_at',
        'composition',
        'customer_description',
        'testing_comments',
        'colour_approval_comments',
    ];

    // protected function Name(): Attribute
    // {
    //     return Attribute::make(
    //         get: fn ($value) => Str::title($value),
    //     );
    // }

    // public function saveWithUser(){
    //     $this->last_updated_by = auth()->user()->id;
    //     $this->save();
    // }

	public function style_versions()
    {
        return $this->belongsTo(StyleVersions::class);
    }
	public function colourway_yarns()
    {
        return $this->hasMany(ColourwayYarns::class);
    }
    // public function colourway_accessories() //REMOVE AFTER UPGRADE
    // {
    //     return $this->hasMany(ColourwayAccessories::class);
    // }
    public function samples()
    {
	    return $this->hasMany(Samples::class);
    }
    public function customer_order_lines()
    {
	    return $this->hasMany(CustomerOrderLines::class);
    }

    public function shipment_lines()
    {
        return $this->hasMany(ShipmentLines::class);
    }

    // public function getIsCancelledAttribute(){
    //     if($this->cancelled || $this->style_versions->styles->cancelled)
    //         return TRUE;
    // }

	// public function scopeSearch($query, $value){
	// 	$query
	// 	->whereRelation('style_versions', 'styles.customer_ref', 'like', "%{$value}%")
	// 	->orwhereRelation('style_versions.styles', 'designs_id', 'like', "%{$value}%")
	// 	->orwhereRelation('style_versions.styles', 'customers_id', 'like', "%{$value}%")
	// 	->orwhereRelation('style_versions.styles.designs', 'description', 'like', "%{$value}%");
	// }

    public function scopeSearch($query, $value)
    {
        $sanitizedValue = preg_replace('/[^a-zA-Z0-9\s-]/', '', $value);
        // Split the search string into multiple terms
        $terms = preg_split('/\s+/', trim($value));

        $query->where(function($query) use ($terms) {
            foreach ($terms as $term) {
                // For each term, require at least one matching condition
                $query->where(function($q) use ($term) {
                    $q->whereRelation('style_versions', 'styles.customer_ref', 'like', "%{$term}%")
                    ->orwhereRelation('style_versions.factories', 'name', 'like', "%{$term}%")
                    ->orwhereRelation('style_versions.styles', 'designs_id', 'like', "%{$term}%")
                    ->orwhereRelation('style_versions.styles', 'customers_id', 'like', "%{$term}%")
                    ->orwhereRelation('style_versions.styles.designs', 'description', 'like', "%{$term}%");
                });
            }
        });
    }

    public function getYarns(){
        $yarns = [];
            foreach($this->colourway_yarns as $yarn){
                $yarns[] = ['colour_id' => $yarn->yarn_colours->id, 'count' => $yarn->yarn_colours->yarn->counts->count, 'description' => $yarn->yarn_colours->yarn->description, 'colour' => $yarn->yarn_colours->reference . ' ' . $yarn->yarn_colours->description];
            }
        $collection = collect($yarns);

        $unique = $collection->unique(function ($item) {
            return $item['description'].$item['colour'];
        });

        return $unique;
    }

    public function getYarnCostKgBaseAttribute(){
        $cost = 0;
        // foreach($this->load('colourway_yarns.yarn_colours.yarn.suppliers', 'style_versions.styles.seasons')->colourway_yarns as $cy){
        foreach($this->colourway_yarns as $cy){
            $cost = $cost + Conversions::convertCurrency($cy->yarn_colours->yarn->suppliers->currency, '£', $cy->price * ($cy->percentage/100), $this->style_versions->styles->seasons);
        }
        return $cost;
    }

    public function getPriceModelAttribute(){
        return $this->price();
    }

    public function price(?int $sizeId = null, ?int $phaseId = null)
    {
        // dd($this->style_versions_id);
        $cacheKey = sprintf(
            'price:v2:sv_%d:cw_%d:size_%s:phase_%s',
            $this->style_versions_id,
            $this->id,
            $sizeId ?? 'null',
            $phaseId ?? 'null'
        );

        // Get season ID for cache tagging
        $seasonId = $this->style_versions->styles->seasons_id ?? null;
        $cacheTags = [
            'prices',
            'colourway:'.$this->id,
            'styleversion:'.$this->style_versions_id,
        ];
        
        if ($seasonId) {
            $cacheTags[] = 'season:'.$seasonId;
        }

        return Cache::tags($cacheTags)
        ->remember($cacheKey, now()->addWeek(), function () use ($sizeId, $phaseId) {

            // Check if *any* price exists for this style version (including NULL colourways)
            $anyPriceExists = Price::where('style_versions_id', $this->style_versions_id)
                ->where(function ($query) {
                    $query->where('colourways_id', $this->id)
                        ->orWhereNull('colourways_id'); // Include prices without a specific colourway
                })
                ->exists();


            // Check if a *general* (non-size-specific) price exists (including NULL colourways)
            $hasGeneralPrice = Price::where('style_versions_id', $this->style_versions_id)
            ->where(function ($query) {
                $query->where('colourways_id', $this->id)
                ->orWhereNull('colourways_id'); // Include prices without a specific colourway
            })
            ->where('sizes_id', 1)
            ->exists();

            // dd($anyPriceExists, $hasGeneralPrice, !$hasGeneralPrice && $sizeId === null);

            if (!$anyPriceExists) {
                return collect([
                    'price' => null,
                    'message' => 'No price exists for this colourway or style version.',
                    'status' => 'no_price',
                ]);
            }

            // if (!$hasGeneralPrice && $sizeId === null) {
            //     return collect([
            //         'price' => null,
            //         'message' => 'Prices exist, but they are size-specific. Please select a size.',
            //         'status' => 'size_required',
            //     ]);
            // }

            // Fetch the best price
            $priceQuery = Price::with([
                'style_versions',
                'style_versions.factories:id,currency,customs',
                'style_versions.styles:id,customers_id,seasons_id',
                'style_versions.styles.customers:id,name,currency,customs',
                'style_versions.styles.customers.customer_payment_terms:customer_id,percentage,days',
            ])
            ->where('style_versions_id', $this->style_versions_id)
            ->where(function ($query) {
                $query->where('colourways_id', $this->id)
                    ->orWhereNull('colourways_id');
            })
            ->when($phaseId, function ($query) use ($phaseId) {
                $query->where(function ($q) use ($phaseId) {
                    $q->where('phase_id', $phaseId)
                      ->orWhereNull('phase_id');
                });
            })
            ->when($sizeId, function ($query) use ($sizeId) {
                $query->where(function ($q) use ($sizeId) {
                    $q->where('sizes_id', $sizeId)
                    ->orWhere('sizes_id', 1)
                    ->orWhereNull('sizes_id');
                });
            })
            ->orderByRaw("CASE WHEN quote_status = 'confirmed' THEN 1 ELSE 0 END DESC")
            ->orderByRaw(
                "CASE WHEN colour_type = ? THEN 3 WHEN colour_type IS NULL THEN 2 ELSE 1 END DESC",
                [$this->colour_type]
            )
            ->orderByRaw(
                "CASE WHEN colourways_id = ? THEN 2 WHEN colourways_id IS NULL THEN 1 ELSE 0 END DESC",
                [$this->id]
            )
            ->when($phaseId, function ($query) use ($phaseId) {
                $query->orderByRaw(
                    "CASE WHEN phase_id = ? THEN 2 WHEN phase_id IS NULL THEN 1 ELSE 0 END DESC",
                    [$phaseId]
                );
            })
            ->when($sizeId, function ($query) use ($sizeId) {
                $query->orderByRaw(
                    "CASE WHEN sizes_id = ? THEN 3 WHEN sizes_id = 1 THEN 2 WHEN sizes_id IS NULL THEN 1 ELSE 0 END DESC",
                    [$sizeId]
                );
            })
            ->orderByDesc('min_qty')
            ->limit(1);

            // \Log::debug($priceQuery->toSql(), $priceQuery->getBindings());

            $price = $priceQuery->first();

            if($price){
                return collect([
                    'price' => $price,
                    'message' => 'Price found successfully.',
                    'status' => 'success',
                ]);
            }
            else{
                return collect([
                    'price' => NULL,
                    'message' => 'Price not found.',
                    'status' => 'no_price',
                ]);
            }
        });
    }












    // public function getYarnsNoColour(){
    //     $yarns = [];
    //     foreach($this->colourway_yarns as $yarn){
    //         $yarns[] = ['count' => $yarn->yarn_colours->yarn->counts->count, 'description' => $yarn->yarn_colours->yarn->description,];
    //     }
    //     $collection = collect($yarns);

    //     $unique = $collection->unique(function ($item) {
    //         return $item['description'];
    //     });

    //     return $unique;
    // }


    // public function composition(){
    //     $composition = [];
    //     foreach($this->colourway_yarns as $yarnLine){
    //         foreach($yarnLine->yarns->yarn_compositions as $material){
    //             if(!($composition[$material->materials_id] ?? null)){ //if array key exists
    //                 $composition[$material->materials_id] = ['material' => 0, 'percentage' => 0];
    //             }
    //             $composition[$material->materials_id]['material'] = $material->materials->material;
    //             $composition[$material->materials_id]['percentage'] = $composition[$material->materials_id]['percentage'] + (($material->percentage/100) * ($yarnLine->percentage/100) * 100);
    //         }
    //     }
	// 	$composition = collect($composition);
	// 	return $composition;
    // }

    // public function sampleDetails($sampleType){
    //     if($this->cancelled){
    //         return NULL;
    //     }
    //     elseif(($sampleType == 1 || $sampleType == 2 || $sampleType == 3 || $sampleType == 4) && $this->style_versions->styles->carryover == TRUE){
    //         return NULL;
    //     }
    //     //dd($this->samples);
    //     elseif($this->samples->where('sample_types_id', $sampleType)->where('accepted', TRUE)->isEmpty() != TRUE){
    //         return 'Approved';
    //     }
    //     elseif($this->samples->where('sample_types_id', $sampleType)->isEmpty() != TRUE){
    //         if($this->samples->sortBy('created_at')->first()->date_sent == NULL)
    //             return 'Expected ' . date_format(new DateTime($this->samples->sortBy('created_at')->first()->date_expected), 'd-M');
    //         else
    //             return 'Sent To Customer';
    //     }
    //     else{
    //         foreach($this->style_versions->styles->customers->setting('samplesRequired') as $sample){
    //             if($sample == $sampleType)
    //                 return 'Required';
    //         }
    //     }
    // }

    // public function composition(){
	// 	$composition = [];
    //     foreach($this->colourway_yarns as $yarnLine){
    //         if(!empty($yarnLine->yarns)){
    //             foreach($yarnLine->yarns->yarn_compositions as $material){
    //                 if(!($composition[$material->materials_id] ?? null)){ //if array key exists
    //                     $composition[$material->materials_id] = ['material' => 0, 'percentage' => 0];
    //                 }
    //                 $composition[$material->materials_id]['material'] = $material->materials->material;
    //                 $composition[$material->materials_id]['percentage'] = $composition[$material->materials_id]['percentage'] + (($material->percentage/100) * ($yarnLine->percentage/100) * 100);
    //             }
    //         }
    //     }
	// 	$composition = collect($composition);
	// 	return $composition;
    // }

    public function getImageUrlAttribute(){
        if($this->image)
            return url(asset('storage/' . $this->image));
    }
    public function getThumbUrlAttribute(){
        if($this->img_thumb)
            return url(asset('storage/' . $this->img_thumb));
    }

    protected static function invalidatePriceResolutions($colourway)
    {
        try {
            $service = app(\App\Services\PriceResolutionService::class);
            $service->invalidateByColourway($colourway->id);
        } catch (\Exception $e) {
            \Log::error('Failed to invalidate price resolutions for colourway', [
                'colourway_id' => $colourway->id,
                'error' => $e->getMessage()
            ]);
        }
    }
}
