<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Season;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

class SeasonController extends Controller
{
    public function index()
    {
        $seasons = Season::orderByDesc('year')->orderBy('half')->paginate(20);
        return view('admin.seasons.index', compact('seasons'));
    }

    public function create()
    {
        return view('admin.seasons.create');
    }

    public function store(Request $request)
    {
        $data = $request->validate([
            'code' => 'required|string|max:10|unique:seasons,code',
            'half' => 'required|in:AW,SS',
            'year' => 'required|integer|min:2000|max:2100',
            'label' => 'nullable|string|max:60',
            'description' => 'nullable|string',
            'image' => 'nullable|image|max:4096',
            'starts_on' => 'nullable|date',
            'ends_on' => 'nullable|date|after_or_equal:starts_on',
            'is_active' => 'boolean',
        ]);

        $data['is_active'] = $request->boolean('is_active');
        // Generate label if not provided
        if (empty($data['label'])) {
            $data['label'] = ($data['half'] === 'AW' ? 'Autumn/Winter' : 'Spring/Summer') . ' ' . $data['year'];
        }
        if ($request->hasFile('image')) {
            $path = $request->file('image')->store('seasons', 'public');
            // Store a web-accessible path so existing views continue to work
            $data['image_path'] = Storage::url($path);
        }
        // Remove temporary field from $data if present
        unset($data['image']);
        Season::create($data);

        return redirect()->route('admin.seasons.index')->with('status', 'Season created');
    }

    public function edit(Season $season)
    {
        return view('admin.seasons.edit', compact('season'));
    }

    public function update(Request $request, Season $season)
    {
        $data = $request->validate([
            'code' => 'required|string|max:10|unique:seasons,code,'.$season->id,
            'half' => 'required|in:AW,SS',
            'year' => 'required|integer|min:2000|max:2100',
            'label' => 'nullable|string|max:60',
            'description' => 'nullable|string',
            'image' => 'nullable|image|max:4096',
            'starts_on' => 'nullable|date',
            'ends_on' => 'nullable|date|after_or_equal:starts_on',
            'is_active' => 'boolean',
        ]);

        $data['is_active'] = $request->boolean('is_active');
        if (empty($data['label'])) {
            $data['label'] = ($data['half'] === 'AW' ? 'Autumn/Winter' : 'Spring/Summer') . ' ' . $data['year'];
        }
        if ($request->hasFile('image')) {
            $path = $request->file('image')->store('seasons', 'public');
            $data['image_path'] = Storage::url($path);
        }
        unset($data['image']);
        $season->update($data);

        return redirect()->route('admin.seasons.index')->with('status', 'Season updated');
    }

    public function destroy(Season $season)
    {
        $season->delete();
        return redirect()->route('admin.seasons.index')->with('status', 'Season deleted');
    }
}


