<?php

namespace App\Http\Controllers;

use App\Models\Season;
use App\Models\Category;
use App\Models\Item;
use Illuminate\Http\Request;

class FrontendController extends Controller
{
    public function home()
    {
        $seasons = Season::where('is_active', true)
            ->orderBy('sort_order')
            ->take(3)
            ->get();

        $categories = Category::where('is_active', true)
            ->orderBy('sort_order')
            ->get();

        $featuredProducts = Item::where('is_active', true)
            ->where('featured', true)
            ->with(['season', 'category', 'colourways'])
            ->take(6)
            ->get();

        // If no featured products, get the latest ones
        if ($featuredProducts->isEmpty()) {
            $featuredProducts = Item::where('is_active', true)
                ->with(['season', 'category', 'colourways'])
                ->latest()
                ->take(6)
                ->get();
        }

        return view('frontend.home', compact('seasons', 'categories', 'featuredProducts'));
    }

    public function collections()
    {
        $seasons = Season::where('is_active', true)
            ->orderBy('sort_order')
            ->withCount(['items' => function ($query) {
                $query->where('is_active', true);
            }])
            ->get();

        return view('frontend.collections', compact('seasons'));
    }

    public function collectionsBySeason($slug)
    {
        $season = Season::where('slug', $slug)
            ->where('is_active', true)
            ->firstOrFail();

        $categories = Category::where('is_active', true)
            ->orderBy('sort_order')
            ->withCount(['items' => function ($query) use ($season) {
                $query->where('is_active', true)
                    ->where('seasons_id', $season->id);
            }])
            ->having('items_count', '>', 0)
            ->get();

        return view('frontend.collections-season', compact('season', 'categories'));
    }

    public function category($slug)
    {
        $category = Category::where('slug', $slug)
            ->where('is_active', true)
            ->firstOrFail();

        $products = Item::where('categories_id', $category->id)
            ->where('is_active', true)
            ->with(['season', 'colourways'])
            ->paginate(12);

        $seasons = Season::where('is_active', true)
            ->orderBy('sort_order')
            ->get();

        return view('frontend.category', compact('category', 'products', 'seasons'));
    }

    public function product($slug)
    {
        $product = Item::where('slug', $slug)
            ->where('is_active', true)
            ->with(['season', 'category', 'colourways'])
            ->firstOrFail();

        // Get related products from the same category
        $relatedProducts = Item::where('categories_id', $product->categories_id)
            ->where('id', '!=', $product->id)
            ->where('is_active', true)
            ->with(['season', 'colourways'])
            ->take(4)
            ->get();

        return view('frontend.product', compact('product', 'relatedProducts'));
    }

    public function search(Request $request)
    {
        $query = $request->get('q');
        
        $products = Item::where('is_active', true)
            ->where(function ($q) use ($query) {
                $q->where('name', 'LIKE', "%{$query}%")
                    ->orWhere('description', 'LIKE', "%{$query}%")
                    ->orWhere('short_description', 'LIKE', "%{$query}%")
                    ->orWhere('sku', 'LIKE', "%{$query}%");
            })
            ->with(['season', 'category', 'colourways'])
            ->paginate(12);

        return view('frontend.search', compact('products', 'query'));
    }

    // Documentation Pages
    public function about()
    {
        return view('frontend.about');
    }

    public function careGuide()
    {
        return view('frontend.care-guide');
    }

    public function patterns()
    {
        return view('frontend.patterns');
    }

    public function contact()
    {
        return view('frontend.contact');
    }

    // Legal Pages
    public function terms()
    {
        return view('frontend.legal.terms');
    }

    public function privacy()
    {
        return view('frontend.legal.privacy');
    }

    public function cookies()
    {
        return view('frontend.legal.cookies');
    }
}

