const path = require('path');
const fs = require('fs');
const fsPromises = require('fs').promises;
require('dotenv').config();
const express = require('express');
const cookieParser = require('cookie-parser');
const multer = require('multer');
const { customAlphabet } = require('nanoid');
const sharp = require('sharp');
const helmet = require('helmet');
const rateLimit = require('express-rate-limit');

// Multi-tenant modules
const db = require('./db-multi-tenant');
const storage = require('./storage');
const auth = require('./auth');
const tenant = require('./tenant');
const superuser = require('./superuser');
const stripeService = require('./stripe-service');

const app = express();
const PORT = process.env.PORT || 3000;

// Trust the first proxy (e.g., GCP/AWS load balancer)
app.set('trust proxy', 1);

app.set('view engine', 'ejs');
app.set('views', path.join(__dirname, '..', 'views'));

// Security headers
app.use(helmet({
  contentSecurityPolicy: {
    directives: {
      defaultSrc: ["'self'"],
      styleSrc: ["'self'", "'unsafe-inline'", "https://fonts.googleapis.com"],
      fontSrc: ["'self'", "https://fonts.gstatic.com"],
      imgSrc: ["'self'", "data:", "https://storage.googleapis.com", "https://storage.cloud.google.com"],
      scriptSrc: ["'self'", "'unsafe-inline'", "https://js.stripe.com"],
      frameSrc: ["https://js.stripe.com", "https://hooks.stripe.com"],
      connectSrc: ["'self'", "https://api.stripe.com"]
    }
  },
  crossOriginEmbedderPolicy: false
}));

// Rate limiters
const generalLimiter = rateLimit({
  windowMs: 15 * 60 * 1000,
  max: 200,
  message: 'Too many requests, please try again later.',
  standardHeaders: true,
  legacyHeaders: false,
  skip: (req) => {
    if (req.method === 'GET' || req.method === 'HEAD') {
      const p = req.path || '';
      if (
        p.startsWith('/thumb/') ||
        p.startsWith('/uploads/') ||
        /\.(css|js|png|jpg|jpeg|gif|webp|svg|ico|woff2?|map)$/i.test(p)
      ) {
        return true;
      }
    }
    return false;
  }
});

const authLimiter = rateLimit({
  windowMs: 15 * 60 * 1000,
  max: 5,
  message: 'Too many authentication attempts, please try again later.',
  standardHeaders: true,
  legacyHeaders: false
});

const uploadLimiter = rateLimit({
  windowMs: 60 * 60 * 1000,
  max: Number(process.env.UPLOADS_PER_HOUR || 500),
  message: 'Too many uploads, please try again later.',
  standardHeaders: true,
  legacyHeaders: false
});

app.use(generalLimiter);
app.use(express.urlencoded({ extended: true, limit: '1mb' }));
app.use(express.json({ limit: '1mb' }));
app.use(cookieParser());
app.use(express.static(path.join(__dirname, '..', 'public')));
app.use('/uploads', express.static(path.join(__dirname, '..', 'uploads')));

// Helper to convert hex to RGB for CSS variables
function hexToRgb(hex) {
  const shorthandRegex = /^#?([a-f\d])([a-f\d])([a-f\d])$/i;
  hex = hex.replace(shorthandRegex, function(m, r, g, b) {
    return r + r + g + g + b + b;
  });
  const result = /^#?([a-f\d]{2})([a-f\d]{2})([a-f\d]{2})$/i.exec(hex);
  return result ? `${parseInt(result[1], 16)}, ${parseInt(result[2], 16)}, ${parseInt(result[3], 16)}` : null;
}

// Apply authentication middleware to all routes
app.use(auth.authenticateUser);

// Apply tenant detection middleware to all routes
app.use(tenant.detectTenant);

// Make storage helper and default siteConfig available to all views
app.use((req, res, next) => {
  res.locals.storage = storage;
  res.locals.baseDomain = tenant.BASE_DOMAIN;
  
  // Ensure siteConfig is always set (for app subdomain routes that don't have a tenant)
  if (!res.locals.siteConfig) {
    // For authenticated users on app subdomain, use their profile settings
    if (req.user) {
      res.locals.siteConfig = {
        siteName: req.user.siteName || req.user.username || 'Vybe Photo',
        primaryColor: req.user.primaryColor || '#3b82f6',
        logoUrl: req.user.logoUrl || '/nrw-web.png',
        hotChocolateDefaultUrl: req.user.hotChocolateDefaultUrl || 'https://square.link/u/NkgDiQCk',
        hotChocolateText: req.user.hotChocolateText || 'Like these pics? Buy me a Hot Chocolate',
        contactEmail: req.user.contactEmail || '',
        socialLinks: req.user.socialLinks || {},
        ageVerificationEnabled: req.user.requireAgeVerification || false,
        primaryColorRgb: hexToRgb(req.user.primaryColor || '#3b82f6')
      };
    } else {
      // Default Vybe Photo branding for non-authenticated app subdomain pages
      res.locals.siteConfig = {
        siteName: 'Vybe Photo',
        primaryColor: '#3b82f6',
        logoUrl: '/nrw-web.png',
        hotChocolateDefaultUrl: 'https://square.link/u/NkgDiQCk',
        hotChocolateText: 'Like these pics? Buy me a Hot Chocolate',
        contactEmail: '',
        socialLinks: {},
        ageVerificationEnabled: false,
        primaryColorRgb: hexToRgb('#3b82f6')
      };
    }
  }
  
  next();
});

// Nanoid code generator (A-Z 0-9, 8 chars)
const alphabet = 'ABCDEFGHJKLMNPQRSTUVWXYZ23456789';
const nanoShort = customAlphabet(alphabet, 8);

// Input validation helpers
function validateTitle(title) {
  return title && typeof title === 'string' && title.trim().length > 0 && title.length <= 200;
}

function validateDescription(description) {
  return !description || (typeof description === 'string' && description.length <= 1000);
}

function validateUrl(url) {
  if (!url) return true;
  try {
    new URL(url);
    return url.length <= 500;
  } catch {
    return false;
  }
}

// Generate thumbnail for an image
async function generateThumbnail(userId, albumId, filename) {
  try {
    await storage.generateAndUploadThumbnail(userId, albumId, filename);
    return true;
  } catch (error) {
    console.error('Error generating thumbnail:', error);
    return false;
  }
}

// Age verification middleware for public routes
function requireAgeVerification(req, res, next) {
  if (!req.tenant || !req.tenant.require_age_verification) {
    return next();
  }
  
  if (req.cookies.age_verified === 'true') {
    return next();
  }
  
  res.locals.showAgeGate = true;
  res.locals.originalUrl = req.originalUrl;
  next();
}

// ============================================================================
// MAIN SITE ROUTES (www.vybephoto.com or base domain)
// ============================================================================

// Homepage routing
app.get('/', (req, res) => {
  // If this is a tenant subdomain (username.vybephoto.com), show their gallery
  if (req.tenant) {
    return handleTenantHomepage(req, res);
  }
  
  // If this is the app subdomain (app.vybephoto.com), show login/register page
  if (req.tenantInfo.isAppSubdomain) {
    return showAppLandingPage(req, res);
  }
  
  // If this is www or base domain, show message about Joomla site
  res.send(`
    <!DOCTYPE html>
    <html lang="en">
    <head>
      <meta charset="UTF-8">
      <meta name="viewport" content="width=device-width, initial-scale=1.0">
      <title>Vybe Photo</title>
      <style>
        body {
          font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, Oxygen, Ubuntu, Cantarell, sans-serif;
          margin: 0;
          background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
          min-height: 100vh;
          display: flex;
          align-items: center;
          justify-content: center;
          color: white;
          text-align: center;
          padding: 20px;
        }
        .container { max-width: 600px; }
        h1 { font-size: 48px; margin: 0 0 16px 0; }
        p { font-size: 20px; margin: 0 0 24px 0; opacity: 0.9; line-height: 1.6; }
        .btn {
          display: inline-block;
          padding: 16px 32px;
          margin: 8px;
          background: white;
          color: #667eea;
          text-decoration: none;
          border-radius: 8px;
          font-weight: 600;
          font-size: 18px;
        }
        code { 
          background: rgba(255,255,255,0.2); 
          padding: 4px 8px; 
          border-radius: 4px;
          font-family: 'Courier New', monospace;
        }
      </style>
    </head>
    <body>
      <div class="container">
        <h1>📸 Vybe Photo</h1>
        <p>This is the Node.js application backend.</p>
        <p>The main website should be served by Joomla at this domain.</p>
        <p>For the photographer portal, visit:<br>
          <code>app.${tenant.BASE_DOMAIN}</code>
        </p>
        <a href="http://app.${tenant.BASE_DOMAIN}:${PORT}" class="btn">Go to Photographer Portal</a>
      </div>
    </body>
    </html>
  `);
});

// App subdomain landing page (app.vybephoto.com)
function showAppLandingPage(req, res) {
  // If user is already logged in, redirect to admin portal
  if (req.user) {
    return res.redirect('/admin');
  }
  
  // Default primary color for app subdomain
  const primaryColor = '#3b82f6';
  const primaryColorRgb = '59, 130, 246';
  
  // Show marketing/landing page with login and register options
  res.send(`
    <!DOCTYPE html>
    <html lang="en">
    <head>
      <meta charset="UTF-8">
      <meta name="viewport" content="width=device-width, initial-scale=1.0">
      <title>Vybe Photo - Professional Photo Galleries for Photographers</title>
      <link rel="preconnect" href="https://fonts.googleapis.com" />
      <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin />
      <link href="https://fonts.googleapis.com/css2?family=Rubik:wght@400;500;600;700;800&display=swap" rel="stylesheet" />
      <style>
        :root { --primary-color: ${primaryColor}; --primary-color-rgb: ${primaryColorRgb}; }
        body {
          font-family: 'Rubik', -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
          margin: 0;
          background: radial-gradient(ellipse 1400px 900px at 30% -15%, ${primaryColor}1f, transparent 50%), radial-gradient(circle 600px at 80% 120%, ${primaryColor}0f, transparent 60%), #09090b;
          min-height: 100vh;
          color: #fafafa;
        }
        .container { 
          max-width: 1200px; 
          margin: 0 auto;
          padding: 80px 24px;
        }
        .header {
          text-align: center;
          margin-bottom: 64px;
        }
        h1 { 
          font-size: clamp(2.5rem, 6vw, 4rem); 
          margin: 0 0 16px 0; 
          font-weight: 700;
          letter-spacing: -0.02em;
          color: #fafafa;
        }
        .tagline { 
          font-size: clamp(1.125rem, 2vw, 1.5rem); 
          margin: 0 0 40px 0; 
          color: #a1a1aa;
          font-weight: 400;
        }
        .cta-section {
          display: flex;
          flex-direction: column;
          align-items: center;
          gap: 16px;
          margin-bottom: 48px;
        }
        .btn {
          display: inline-block;
          padding: 14px 28px;
          background: linear-gradient(135deg, ${primaryColor}33 0%, ${primaryColor}1f 100%);
          border: 1px solid ${primaryColor}80;
          color: ${primaryColor};
          text-decoration: none;
          border-radius: 8px;
          font-weight: 600;
          font-size: 1rem;
          transition: all 0.2s;
          box-shadow: 0 2px 8px ${primaryColor}26, inset 0 1px 0 rgba(255,255,255,0.2);
        }
        .btn:hover {
          border-color: ${primaryColor}cc;
          background: linear-gradient(135deg, ${primaryColor}4d 0%, ${primaryColor}2e 100%);
          box-shadow: 0 4px 16px ${primaryColor}40, 0 0 30px ${primaryColor}26, inset 0 1px 0 rgba(255,255,255,0.3);
          transform: translateY(-1px);
        }
        .btn-outline {
          background: rgba(255,255,255,0.03);
          border: 1px solid rgba(255,255,255,0.1);
          color: #d4d4d8;
        }
        .btn-outline:hover {
          background: rgba(255,255,255,0.06);
          border-color: rgba(255,255,255,0.2);
          color: #fafafa;
        }
        .gallery-link {
          margin-top: 24px;
          padding-top: 24px;
          border-top: 1px solid rgba(255,255,255,0.06);
        }
        .gallery-link p {
          font-size: 0.9rem;
          color: #71717a;
          margin: 0 0 12px 0;
        }
        .features {
          display: grid;
          grid-template-columns: repeat(auto-fit, minmax(220px, 1fr));
          gap: 24px;
          margin-bottom: 64px;
        }
        .feature { 
          padding: 32px;
          background: rgba(24,24,27,0.6);
          border: 1px solid rgba(255,255,255,0.06);
          border-radius: 12px;
          text-align: center;
          transition: all 0.2s;
        }
        .feature:hover {
          border-color: rgba(${primaryColorRgb}, 0.2);
          box-shadow: 0 8px 24px rgba(0,0,0,0.5), 0 0 20px rgba(${primaryColorRgb}, 0.15);
        }
        .feature-icon { 
          width: 48px;
          height: 48px;
          margin: 0 auto 16px;
          display: block;
          color: ${primaryColor};
          opacity: 0.9;
        }
        .feature h3 {
          font-size: 1.125rem;
          font-weight: 600;
          margin: 0 0 8px 0;
          color: #fafafa;
        }
        .feature p {
          font-size: 0.875rem;
          color: #71717a;
          margin: 0;
        }
        .pricing {
          display: grid;
          grid-template-columns: repeat(auto-fit, minmax(280px, 1fr));
          gap: 24px;
        }
        .price-card {
          background: rgba(24,24,27,0.6);
          border: 1px solid rgba(255,255,255,0.06);
          padding: 32px;
          border-radius: 12px;
          transition: all 0.2s;
        }
        .price-card:hover {
          border-color: rgba(${primaryColorRgb}, 0.3);
          box-shadow: 0 8px 24px rgba(0,0,0,0.5), 0 0 20px rgba(${primaryColorRgb}, 0.15);
        }
        .price-card h3 { 
          margin: 0 0 16px 0; 
          font-size: 1.5rem; 
          font-weight: 700;
          color: ${primaryColor};
        }
        .price { 
          font-size: 2rem; 
          font-weight: 700; 
          margin: 16px 0; 
          color: #fafafa;
        }
        .price small {
          font-size: 1rem;
          font-weight: 400;
          color: #71717a;
        }
        .price-features { 
          list-style: none; 
          padding: 0; 
          margin: 24px 0 0 0; 
        }
        .price-features li { 
          padding: 10px 0; 
          border-bottom: 1px solid rgba(255,255,255,0.06);
          color: #d4d4d8;
          font-size: 0.9rem;
        }
        .price-features li:last-child { border-bottom: none; }
      </style>
    </head>
    <body>
      <div class="container">
        <div class="header">
          <h1 style="display: flex; align-items: center; justify-content: center; gap: 12px;">
            <svg width="48" height="48" viewBox="0 0 24 24" fill="none" stroke="${primaryColor}" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" style="flex-shrink: 0;">
              <rect x="3" y="3" width="18" height="18" rx="2" ry="2"></rect>
              <circle cx="8.5" cy="8.5" r="1.5"></circle>
              <polyline points="21 15 16 10 5 21"></polyline>
            </svg>
            <span>Vybe Photo</span>
          </h1>
          <p class="tagline">Beautiful photo galleries for professional photographers</p>
          <p style="font-size: clamp(1rem, 2vw, 1.125rem); color: #d4d4d8; margin: 24px auto 0; max-width: 600px; line-height: 1.6;">
            Simple, intuitive, and powerful. Upload your photos, share with clients, and let us handle the rest. No technical knowledge required.
          </p>
          <div class="cta-section">
            <div style="display: flex; gap: 12px; flex-wrap: wrap; justify-content: center;">
              <a href="/register?plan=starter" class="btn">Start Free 14-Day Trial</a>
              <a href="/login" class="btn btn-outline">Sign In</a>
            </div>
            <div class="gallery-link">
              <p>Already have a gallery?</p>
              <a href="http://${tenant.BASE_DOMAIN}" class="btn btn-outline" style="font-size: 0.9rem; padding: 10px 20px;">View Gallery Section</a>
            </div>
          </div>
        </div>
        
        <div class="features">
          <div class="feature">
            <svg class="feature-icon" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
              <path d="M12 2L2 7l10 5 10-5-10-5z"></path>
              <path d="M2 17l10 5 10-5"></path>
              <path d="M2 12l10 5 10-5"></path>
            </svg>
            <h3>Custom Branding</h3>
            <p>Your logo, colors, domain</p>
          </div>
          <div class="feature">
            <svg class="feature-icon" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
              <rect x="3" y="11" width="18" height="11" rx="2" ry="2"></rect>
              <path d="M7 11V7a5 5 0 0 1 10 0v4"></path>
            </svg>
            <h3>Private & Public Albums</h3>
            <p>Access codes & password protection</p>
          </div>
          <div class="feature">
            <svg class="feature-icon" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
              <circle cx="12" cy="12" r="10"></circle>
              <path d="M12 6v6l4 2"></path>
            </svg>
            <h3>Simple & Intuitive</h3>
            <p>Easy to use, no learning curve</p>
          </div>
          <div class="feature">
            <svg class="feature-icon" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
              <rect x="5" y="2" width="14" height="20" rx="2" ry="2"></rect>
              <line x1="12" y1="18" x2="12.01" y2="18"></line>
            </svg>
            <h3>Mobile-Friendly</h3>
            <p>Beautiful on all devices</p>
          </div>
          <div class="feature">
            <svg class="feature-icon" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
              <polygon points="13 2 3 14 12 14 11 22 21 10 12 10 13 2"></polygon>
            </svg>
            <h3>Lightning Fast</h3>
            <p>Optimized thumbnails</p>
          </div>
        </div>
        
        <div class="pricing">
          <div class="price-card">
            <h3>Starter</h3>
            <div class="price">£10<small>/month</small></div>
            <ul class="price-features">
              <li>✅ 1,000 photos</li>
              <li>✅ 5 GB storage</li>
              <li>✅ Subdomain</li>
              <li>✅ Custom branding</li>
              <li>✅ Unlimited albums</li>
            </ul>
            <div style="margin-top: 24px; padding-top: 24px; border-top: 1px solid rgba(255,255,255,0.06); text-align: center;">
              <a href="/register?plan=starter" class="btn" style="display: inline-block; text-decoration: none;">
                <span>Start Free Trial</span>
              </a>
            </div>
          </div>
          
          <div class="price-card" style="border-color: rgba(${primaryColorRgb}, 0.4); background: linear-gradient(135deg, rgba(${primaryColorRgb}, 0.1) 0%, rgba(${primaryColorRgb}, 0.05) 100%);">
            <h3 style="color: ${primaryColor};">Professional</h3>
            <div class="price">£25<small>/month</small></div>
            <ul class="price-features">
              <li>✅ 10,000 photos</li>
              <li>✅ 50 GB storage</li>
              <li>✅ Custom domain</li>
              <li>✅ Everything in Starter</li>
              <li>✅ Priority support</li>
            </ul>
            <div style="margin-top: 24px; padding-top: 24px; border-top: 1px solid rgba(255,255,255,0.06); text-align: center;">
              <a href="/register?plan=professional" class="btn" style="display: inline-block; text-decoration: none;">
                <span>Get Started</span>
              </a>
            </div>
          </div>
          
          <div class="price-card">
            <h3>Business</h3>
            <div class="price">£75<small>/month</small></div>
            <ul class="price-features">
              <li>✅ 30,000 photos</li>
              <li>✅ 150 GB storage</li>
              <li>✅ Multiple domains</li>
              <li>✅ Everything in Professional</li>
              <li>✅ White-label support</li>
            </ul>
            <div style="margin-top: 24px; padding-top: 24px; border-top: 1px solid rgba(255,255,255,0.06); text-align: center;">
              <a href="/register?plan=business" class="btn" style="display: inline-block; text-decoration: none;">
                <span>Get Started</span>
              </a>
            </div>
          </div>
          
          <div class="price-card">
            <h3 style="display: flex; align-items: center; gap: 8px;">
              <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                <path d="M12 2L2 7l10 5 10-5-10-5z"></path>
                <path d="M2 17l10 5 10-5"></path>
                <path d="M2 12l10 5 10-5"></path>
              </svg>
              Custom Plan
            </h3>
            <div class="price" style="font-size: 1.5rem; color: #d4d4d8;">Custom Pricing</div>
            <ul class="price-features">
              <li>✅ Higher photo limits</li>
              <li>✅ More storage</li>
              <li>✅ Dedicated support</li>
              <li>✅ Custom features</li>
              <li>✅ Tailored to your needs</li>
            </ul>
            <div style="margin-top: 24px; padding-top: 24px; border-top: 1px solid rgba(255,255,255,0.06); text-align: center;">
              <a href="mailto:contact@${tenant.BASE_DOMAIN}" class="btn" style="display: inline-block; text-decoration: none;">
                <span>Contact Us</span>
              </a>
            </div>
          </div>
        </div>
      </div>
    </body>
    </html>
  `);
}

// Middleware to ensure auth routes only work on app subdomain
function requireAppSubdomain(req, res, next) {
  if (!req.tenantInfo.isAppSubdomain) {
    return res.status(404).send('This page is only available at app.' + tenant.BASE_DOMAIN);
  }
  next();
}

// Registration page (accessible on custom domains)
app.get('/register', (req, res) => {
  if (req.user) {
    return res.redirect('/admin');
  }
  
  // Extract and validate plan from query parameter
  const VALID_PLANS = ['starter', 'professional', 'business'];
  const planFromUrl = req.query.plan ? req.query.plan.toLowerCase().trim() : null;
  const selectedPlan = (planFromUrl && VALID_PLANS.includes(planFromUrl)) ? planFromUrl : 'starter';
  
  console.log(`[REGISTER GET] Query param 'plan': ${req.query.plan}, Selected: ${selectedPlan}`);
  
  res.render('auth/register', {
    pageTitle: 'Sign Up',
    baseDomain: tenant.BASE_DOMAIN,
    plan: selectedPlan,
    siteConfig: res.locals.siteConfig || {
      primaryColor: '#3b82f6',
      primaryColorRgb: '59, 130, 246'
    },
    req: req
  });
});

// API endpoint to create checkout session for paid plan registration
app.post('/api/create-registration-checkout', authLimiter, express.json(), async (req, res) => {
  try {
    const { username, email, password, fullName, plan } = req.body;
    
    // Validate input
    if (!username || !email || !password || !plan) {
      return res.status(400).json({ error: 'Missing required fields' });
    }
    
    // Validate plan is paid
    const validPaidPlans = ['professional', 'business'];
    if (!validPaidPlans.includes(plan)) {
      return res.status(400).json({ error: 'This endpoint is only for paid plans' });
    }
    
    // Check if Stripe is configured
    if (!stripeService.isConfigured()) {
      return res.status(500).json({ error: 'Payment system not configured. Please contact support.' });
    }
    
    // Check if username already exists
    const existingUser = await db.getUserByUsername(username);
    if (existingUser) {
      return res.status(400).json({ error: 'Username already taken' });
    }
    
    // Check if email already exists
    const existingEmail = await db.getUserByEmail(email);
    if (existingEmail) {
      return res.status(400).json({ error: 'Email already registered' });
    }
    
    console.log(`[CHECKOUT API] Creating checkout session for ${email} with plan ${plan}`);
    
    // Create checkout session with metadata containing registration info
    const protocol = req.secure || req.headers['x-forwarded-proto'] === 'https' ? 'https' : 'http';
    const baseUrl = `${protocol}://${req.get('host')}`;
    
    // We'll store the registration data in the session metadata
    // After payment, we'll create the user from this metadata
    const sessionParams = {
      mode: 'subscription',
      line_items: [{
        price: stripeService.SUBSCRIPTION_TIERS[plan].priceId,
        quantity: 1,
      }],
      success_url: `${baseUrl}/register/complete?session_id={CHECKOUT_SESSION_ID}`,
      cancel_url: `${baseUrl}/register?plan=${plan}&payment_cancelled=1`,
      client_reference_id: username, // Store username for reference
      customer_email: email,
      subscription_data: {
        metadata: {
          tier: plan,
          username: username,
          fullName: fullName || '',
          registrationPassword: password, // We'll need this to create the account
        },
      },
    };
    
    const session = await stripeService.stripe.checkout.sessions.create(sessionParams);
    
    console.log(`[CHECKOUT API] Created session ${session.id} for ${email}`);
    
    res.json({ url: session.url });
  } catch (error) {
    console.error('[CHECKOUT API] Error creating checkout session:', error);
    res.status(500).json({ error: error.message || 'Failed to create checkout session' });
  }
});

// Handle successful registration checkout
app.get('/register/complete', async (req, res) => {
  try {
    const sessionId = req.query.session_id;
    
    if (!sessionId) {
      return res.redirect('/register?error=invalid_session');
    }
    
    // Retrieve the checkout session
    const session = await stripeService.stripe.checkout.sessions.retrieve(sessionId, {
      expand: ['subscription']
    });
    
    if (!session || session.payment_status !== 'paid') {
      return res.redirect('/register?error=payment_not_completed');
    }
    
    const subscription = session.subscription;
    const metadata = subscription.metadata;
    
    const { username, fullName, registrationPassword, tier } = metadata;
    
    // Check if user already exists (in case of double-processing)
    const existingUser = await db.getUserByUsername(username);
    if (existingUser) {
      // User already created, just log them in
      await auth.loginUser(existingUser.email, registrationPassword, req, res);
      return res.redirect('/admin?welcome=1');
    }
    
    // Create the user account
    const user = await auth.registerUser(
      username,
      session.customer_email,
      registrationPassword,
      fullName,
      tier
    );
    
    // Update with Stripe info
    await db.updateStripeCustomerId(user.id, session.customer);
    await db.updateUser(user.id, {
      stripeSubscriptionId: subscription.id,
      subscriptionStatus: subscription.status,
      subscriptionTier: tier,
    });
    
    console.log(`[REGISTRATION COMPLETE] Created user ${user.id} after payment for plan ${tier}`);
    
    // Log them in
    await auth.loginUser(session.customer_email, registrationPassword, req, res);
    
    res.redirect('/admin?welcome=1');
  } catch (error) {
    console.error('[REGISTRATION COMPLETE] Error:', error);
    res.redirect('/register?error=registration_failed');
  }
});

// Registration handler (accessible on custom domains)
app.post('/register', authLimiter, async (req, res) => {
  try {
    const { username, email, password, fullName, plan } = req.body;
    
    // Validate plan
    const validPlans = ['starter', 'professional', 'business'];
    const subscriptionTier = validPlans.includes(plan) ? plan : 'starter';
    
    const user = await auth.registerUser(username, email, password, fullName, subscriptionTier);
    
    // Log them in automatically
    await auth.loginUser(email, password, req, res);
    
    // For Professional/Business plans, create Stripe Checkout Session for payment
    // For Starter plan, redirect to admin (free trial)
    if (subscriptionTier === 'professional' || subscriptionTier === 'business') {
      try {
        // Check if Stripe is configured
        if (!stripeService.isConfigured()) {
          console.warn('[REGISTRATION] Stripe not configured. Redirecting to subscription page.');
          return res.redirect('/admin/subscription?setup_payment=1&message=Please contact support to set up your subscription');
        }
        
        // Create Stripe customer for the user
        const stripeCustomerId = await stripeService.createCustomer(
          email,
          username,
          fullName || username
        );
        
        // Update user with Stripe customer ID
        await db.updateStripeCustomerId(user.id, stripeCustomerId);
        
        console.log(`[REGISTRATION] Created Stripe customer ${stripeCustomerId} for user ${user.id}`);
        
        // Create Stripe Checkout Session for subscription
        const successUrl = `${req.protocol}://app.${tenant.BASE_DOMAIN}/admin?payment_success=1`;
        const cancelUrl = `${req.protocol}://app.${tenant.BASE_DOMAIN}/register?payment_cancelled=1`;
        
        const checkoutUrl = await stripeService.createCheckoutSession(
          stripeCustomerId,
          subscriptionTier,
          successUrl,
          cancelUrl
        );
        
        console.log(`[REGISTRATION] Redirecting to Stripe Checkout for user ${user.id}`);
        
        // Redirect to Stripe Checkout
        return res.redirect(checkoutUrl);
      } catch (stripeError) {
        console.error('[REGISTRATION] Stripe error:', stripeError);
        // If Stripe fails, redirect to subscription page with error
        return res.redirect('/admin/subscription?setup_payment=1&error=' + encodeURIComponent('Failed to set up payment. Please try again or contact support.'));
      }
    }
    
    // Redirect to admin portal for Starter plan (free trial)
    res.redirect('/admin');
  } catch (error) {
    console.error('Registration error:', error);
    
    // Preserve the selected plan from form submission
    const VALID_PLANS = ['starter', 'professional', 'business'];
    const submittedPlan = req.body.plan ? req.body.plan.toLowerCase().trim() : null;
    const selectedPlan = (submittedPlan && VALID_PLANS.includes(submittedPlan)) ? submittedPlan : 'starter';
    
    res.render('auth/register', {
      pageTitle: 'Sign Up',
      error: error.message,
      formData: req.body,
      plan: selectedPlan,
      baseDomain: tenant.BASE_DOMAIN,
      siteConfig: res.locals.siteConfig || {
        primaryColor: '#3b82f6',
        primaryColorRgb: '59, 130, 246'
      },
      req: req
    });
  }
});

// Login page (accessible on custom domains)
app.get('/login', (req, res) => {
  if (req.user) {
    return res.redirect('/admin');
  }
  res.render('auth/login', {
    pageTitle: 'Sign In',
    returnTo: req.query.returnTo,
    req: req
  });
});

// Login handler (accessible on custom domains)
app.post('/login', authLimiter, async (req, res) => {
  try {
    const { email, password, returnTo } = req.body;
    
    const user = await auth.loginUser(email, password, req, res);
    
    if (!user) {
      return res.render('auth/login', {
        pageTitle: 'Sign In',
        error: 'Invalid email or password',
        formData: req.body,
        req: req
      });
    }
    
    // Redirect to return URL or admin portal
    res.redirect(returnTo || '/admin');
  } catch (error) {
    console.error('Login error:', error);
    res.render('auth/login', {
      pageTitle: 'Sign In',
      error: 'An error occurred during login',
      formData: req.body,
      req: req
    });
  }
});

// Logout
app.get('/logout', async (req, res) => {
  await auth.logoutUser(req, res);
  // Redirect to app subdomain landing page
  res.redirect(`http://app.${tenant.BASE_DOMAIN}`);
});

// Dashboard route removed - redirect to admin portal
app.get('/dashboard', requireAppSubdomain, auth.requireAuth, async (req, res) => {
  res.redirect('/admin');
});

// ============================================================================
// TENANT GALLERY ROUTES (username.vybephoto.com)
// ============================================================================

async function handleTenantHomepage(req, res) {
  const publicAlbums = await db.listPublicAlbums(req.tenant.id);
  const homePageAlbum = await db.getHomePageAlbum(req.tenant.id);
  let homePagePhotos = [];
  
  if (homePageAlbum) {
    homePagePhotos = await db.getPhotosByAlbum(homePageAlbum.id);
  }
  
  res.render('index', {
    pageTitle: 'Access Your Album',
    publicAlbums,
    homePageAlbum,
    homePagePhotos,
    req: req
  });
}

// Age verification endpoint
app.post('/verify-age', tenant.requireTenant, authLimiter, (req, res) => {
  const day = parseInt(req.body.day, 10);
  const month = parseInt(req.body.month, 10);
  const year = parseInt(req.body.year, 10);
  
  if (!day || !month || !year || day < 1 || day > 31 || month < 1 || month > 12 || year < 1900) {
    return res.redirect(req.body.return_to || '/');
  }
  
  const birthDate = new Date(year, month - 1, day);
  const today = new Date();
  let age = today.getFullYear() - birthDate.getFullYear();
  const monthDiff = today.getMonth() - birthDate.getMonth();
  
  if (monthDiff < 0 || (monthDiff === 0 && today.getDate() < birthDate.getDate())) {
    age--;
  }
  
  if (age >= 18) {
    res.cookie('age_verified', 'true', {
      maxAge: 30 * 24 * 60 * 60 * 1000,
      httpOnly: true,
      sameSite: 'strict'
    });
    
    const returnTo = req.body.return_to || '/';
    return res.redirect(returnTo);
  } else {
    return res.redirect('https://www.bbc.co.uk/cbbc');
  }
});

// Access code submission
app.post('/access', tenant.requireTenant, requireAgeVerification, async (req, res) => {
  const code = String(req.body.code || '').trim().toUpperCase();
  if (!code) return res.redirect('/');
  
  // Check if it's a group code first
  const group = await db.getGroupByCode(code, req.tenant.id);
  if (group) {
    return res.redirect(`/g/${encodeURIComponent(code)}`);
  }
  
  // Otherwise, try album
  return res.redirect(`/a/${encodeURIComponent(code)}`);
});

// View album by code
app.get('/a/:code', tenant.requireTenant, requireAgeVerification, async (req, res) => {
  const code = String(req.params.code || '').toUpperCase();
  const album = await db.getAlbumByCode(code);
  
  // Verify album belongs to this tenant
  if (!album || album.user_id !== req.tenant.id) {
    return res.status(404).render('not-found', { pageTitle: 'Album Not Found', req: req });
  }
  
  // Check if album has password protection
  if (album.password_hash) {
    // Check if password is already verified in session
    const verifiedAlbums = req.session.verifiedAlbums || [];
    if (!verifiedAlbums.includes(album.id)) {
      // Show password form
      return res.render('album-password', { 
        pageTitle: 'Enter Password', 
        album: { id: album.id, title: album.title, code: album.code },
        error: req.query.error,
        req: req 
      });
    }
  }
  
  const photos = await db.getPhotosByAlbum(album.id);
  res.render('album', { pageTitle: album.title, album, photos, req: req });
});

// Verify album password
app.post('/a/:code/password', tenant.requireTenant, requireAgeVerification, async (req, res) => {
  const code = String(req.params.code || '').toUpperCase();
  const password = String(req.body.password || '').trim();
  const album = await db.getAlbumByCode(code);
  
  if (!album || album.user_id !== req.tenant.id || !album.password_hash) {
    return res.redirect(`/a/${code}?error=Invalid album`);
  }
  
  // Verify password
  const bcrypt = require('bcrypt');
  const isValid = await bcrypt.compare(password, album.password_hash);
  
  if (!isValid) {
    return res.redirect(`/a/${code}?error=Incorrect password`);
  }
  
  // Store verified album ID in session
  if (!req.session.verifiedAlbums) {
    req.session.verifiedAlbums = [];
  }
  if (!req.session.verifiedAlbums.includes(album.id)) {
    req.session.verifiedAlbums.push(album.id);
  }
  
  res.redirect(`/a/${code}`);
});

// View group by code
app.get('/g/:code', tenant.requireTenant, requireAgeVerification, async (req, res) => {
  const code = String(req.params.code || '').toUpperCase();
  const group = await db.getGroupByCode(code);
  
  // Verify group belongs to this tenant
  if (!group || group.user_id !== req.tenant.id) {
    return res.status(404).render('not-found', { pageTitle: 'Group Not Found', req: req });
  }
  
  const albums = await db.getAlbumsByGroup(group.id);
  res.render('group', { pageTitle: group.name, group, albums, req: req });
});

// Serve or generate thumbnail on-demand
app.get('/thumb/:albumId/:filename', async (req, res) => {
  const albumId = String(req.params.albumId).replace(/[^0-9]/g, '');
  const filename = path.basename(req.params.filename);
  
  if (!albumId || !filename) {
    return res.status(400).send('Invalid request');
  }
  
  // Get album to find user_id
  const album = await db.getAlbumById(albumId);
  if (!album) {
    return res.status(404).send('Album not found');
  }
  
  if (storage.USE_CLOUD_STORAGE) {
    const thumbUrl = storage.getPublicUrl(album.user_id, albumId, filename, true);
    return res.redirect(thumbUrl);
  }
  
  // Local development
  const thumbPath = path.join(__dirname, '..', 'uploads', String(album.user_id), albumId, 'thumbs', filename);
  
  if (fs.existsSync(thumbPath)) {
    return res.sendFile(thumbPath);
  }
  
  const sourcePath = path.join(__dirname, '..', 'uploads', String(album.user_id), albumId, filename);
  
  if (!fs.existsSync(sourcePath)) {
    return res.status(404).send('Image not found');
  }
  
  try {
    await generateThumbnail(album.user_id, albumId, filename);
    res.sendFile(thumbPath);
  } catch (error) {
    console.error('Error serving thumbnail:', error);
    res.sendFile(sourcePath);
  }
});

// ============================================================================
// ADMIN ROUTES (Photographer Dashboard)
// ============================================================================

// Multer storage configured per album
const multerStorage = multer.diskStorage({
  destination: function (req, file, cb) {
    const userId = req.user.id;
    const albumId = req.params.id;
    const albumDir = path.join(__dirname, '..', 'uploads', String(userId), String(albumId));
    fs.mkdirSync(albumDir, { recursive: true });
    cb(null, albumDir);
  },
  filename: function (req, file, cb) {
    const timestamp = Date.now();
    const safeOriginal = path.basename(file.originalname).replace(/[^a-zA-Z0-9._-]/g, '_');
    cb(null, `${timestamp}-${safeOriginal}`);
  }
});

function imageOnlyFilter(req, file, cb) {
  if ((file.mimetype || '').startsWith('image/')) return cb(null, true);
  cb(new Error('Only image uploads are allowed'));
}

const upload = multer({
  storage: multerStorage,
  fileFilter: imageOnlyFilter,
  limits: {
    fileSize: 50 * 1024 * 1024,
    files: 500
  }
});

// Admin dashboard (accessible on custom domains)
app.get('/admin', auth.requireAuth, async (req, res) => {
  const albums = await db.listAlbums(req.user.id);
  const groups = await db.listGroups(req.user.id);
  res.render('admin/index', { 
    pageTitle: 'Admin', 
    albums, 
    groups, 
    galleryUrl: tenant.getGalleryUrl(req.user, req),
    siteConfig: {
      siteName: req.user.siteName || req.user.site_name || req.user.username,
      primaryColor: req.user.primaryColor || req.user.primary_color || '#3b82f6',
      primaryColorRgb: hexToRgb(req.user.primaryColor || req.user.primary_color || '#3b82f6'),
      logoUrl: req.user.logoUrl || req.user.logo_url || '/nrw-web.png',
      hotChocolateDefaultUrl: req.user.hotChocolateDefaultUrl || req.user.hot_chocolate_default_url || 'https://square.link/u/NkgDiQCk',
      hotChocolateText: req.user.hotChocolateText || req.user.hot_chocolate_text || 'Like these pics? Buy me a Hot Chocolate',
      contactEmail: req.user.contactEmail || req.user.contact_email || '',
      socialLinks: req.user.socialLinks || req.user.social_links || {},
      ageVerificationEnabled: req.user.requireAgeVerification || req.user.require_age_verification || false
    },
    req: req 
  });
});

// Subscription page
app.get('/admin/subscription', auth.requireAuth, async (req, res) => {
  const usage = await db.getUserUsageStats(req.user.id);
  const albums = await db.listAlbums(req.user.id);
  
  // Get full user details including created_at
  const fullUser = await db.getUserById(req.user.id);
  
  res.render('admin/subscription', {
    pageTitle: 'Subscription',
    user: {
      ...req.user,
      createdAt: fullUser.created_at
    },
    usage,
    albumCount: albums.length,
    siteConfig: {
      siteName: req.user.siteName || req.user.site_name || req.user.username,
      primaryColor: req.user.primaryColor || req.user.primary_color || '#3b82f6',
      primaryColorRgb: hexToRgb(req.user.primaryColor || req.user.primary_color || '#3b82f6'),
      logoUrl: req.user.logoUrl || req.user.logo_url || '/nrw-web.png'
    }
  });
});

// Change subscription tier
app.post('/admin/subscription/change', auth.requireAuth, async (req, res) => {
  try {
    const newTier = req.body.tier;
    
    // Validate tier
    if (!['starter', 'professional', 'business'].includes(newTier)) {
      return res.status(400).send('Invalid tier');
    }
    
    // Update subscription tier (this will also update limits)
    await db.updateSubscriptionTier(req.user.id, newTier);
    
    // Log the change
    console.log(`User ${req.user.username} changed subscription from ${req.user.subscriptionTier} to ${newTier}`);
    
    res.redirect('/admin/subscription');
  } catch (error) {
    console.error('Error changing subscription:', error);
    res.status(500).send('Failed to change subscription');
  }
});

// Cancel subscription
app.post('/admin/subscription/cancel', auth.requireAuth, async (req, res) => {
  try {
    // Update subscription status to canceled
    await db.pool.query(
      `UPDATE users 
       SET subscription_status = 'canceled',
           subscription_ends_at = CURRENT_TIMESTAMP + INTERVAL '30 days',
           updated_at = CURRENT_TIMESTAMP
       WHERE id = $1`,
      [req.user.id]
    );
    
    // Log the cancellation
    console.log(`User ${req.user.username} canceled their subscription`);
    
    res.redirect('/admin/subscription');
  } catch (error) {
    console.error('Error canceling subscription:', error);
    res.status(500).send('Failed to cancel subscription');
  }
});

// Update email
app.post('/admin/account/update-email', auth.requireAuth, async (req, res) => {
  try {
    const newEmail = String(req.body.email || '').trim().toLowerCase();
    
    if (!newEmail || !/^[^\s@]+@[^\s@]+\.[^\s@]+$/.test(newEmail)) {
      return res.status(400).send('Invalid email address');
    }
    
    // Check if email is already taken
    const existing = await db.getUserByEmail(newEmail);
    if (existing && existing.id !== req.user.id) {
      return res.status(400).send('Email is already in use');
    }
    
    await db.pool.query(
      'UPDATE users SET email = $1, updated_at = CURRENT_TIMESTAMP WHERE id = $2',
      [newEmail, req.user.id]
    );
    
    console.log(`User ${req.user.username} updated email to ${newEmail}`);
    res.redirect('/admin/subscription');
  } catch (error) {
    console.error('Error updating email:', error);
    res.status(500).send('Failed to update email');
  }
});

// Update full name
app.post('/admin/account/update-name', auth.requireAuth, async (req, res) => {
  try {
    const fullName = String(req.body.fullName || '').trim();
    
    await db.pool.query(
      'UPDATE users SET full_name = $1, updated_at = CURRENT_TIMESTAMP WHERE id = $2',
      [fullName || null, req.user.id]
    );
    
    console.log(`User ${req.user.username} updated full name`);
    res.redirect('/admin/subscription');
  } catch (error) {
    console.error('Error updating name:', error);
    res.status(500).send('Failed to update name');
  }
});

// Change password
app.post('/admin/account/change-password', auth.requireAuth, async (req, res) => {
  try {
    const { currentPassword, newPassword } = req.body;
    
    if (!currentPassword || !newPassword) {
      return res.status(400).send('Current and new password are required');
    }
    
    if (newPassword.length < 8) {
      return res.status(400).send('New password must be at least 8 characters');
    }
    
    // Verify current password
    const user = await db.verifyUserPassword(req.user.email, currentPassword);
    if (!user) {
      return res.status(400).send('Current password is incorrect');
    }
    
    // Update password
    await db.updateUserPassword(req.user.id, newPassword);
    
    console.log(`User ${req.user.username} changed their password`);
    res.redirect('/admin/subscription');
  } catch (error) {
    console.error('Error changing password:', error);
    res.status(500).send('Failed to change password');
  }
});

// Payment portal (Stripe customer portal)
app.post('/admin/subscription/payment-portal', auth.requireAuth, async (req, res) => {
  try {
    // For now, show a message about Stripe integration
    // When Stripe is integrated, this will redirect to Stripe's customer portal
    
    if (req.user.stripeCustomerId) {
      // TODO: When Stripe is integrated, create and redirect to customer portal session
      // const stripe = require('stripe')(process.env.STRIPE_SECRET_KEY);
      // const session = await stripe.billingPortal.sessions.create({
      //   customer: req.user.stripeCustomerId,
      //   return_url: `${req.protocol}://${req.get('host')}/admin/subscription`,
      // });
      // return res.redirect(session.url);
      
      return res.send(`
        <html>
          <head>
            <title>Payment Portal</title>
            <style>
              body { 
                font-family: system-ui, -apple-system, sans-serif; 
                background: #09090b; 
                color: white; 
                display: flex; 
                align-items: center; 
                justify-content: center; 
                height: 100vh; 
                margin: 0;
              }
              .card {
                background: rgba(24,24,27,0.95);
                border: 1px solid rgba(255,255,255,0.1);
                border-radius: 12px;
                padding: 48px;
                max-width: 500px;
                text-align: center;
              }
              .btn {
                background: linear-gradient(135deg, #3b82f633 0%, #3b82f61f 100%);
                border: 1px solid #3b82f680;
                color: white;
                padding: 12px 24px;
                border-radius: 8px;
                text-decoration: none;
                display: inline-block;
                margin-top: 24px;
                cursor: pointer;
              }
              .btn:hover {
                background: linear-gradient(135deg, #3b82f64d 0%, #3b82f62e 100%);
              }
            </style>
          </head>
          <body>
            <div class="card">
              <h1 style="margin: 0 0 16px;">💳 Payment Management</h1>
              <p style="color: #a1a1aa; line-height: 1.6;">
                Payment method management through Stripe will be available soon. 
                For now, you can change your subscription plan directly on this page.
              </p>
              <a href="/admin/subscription" class="btn">← Back to Subscription</a>
            </div>
          </body>
        </html>
      `);
    } else {
      return res.send(`
        <html>
          <head>
            <title>Payment Portal</title>
            <style>
              body { 
                font-family: system-ui, -apple-system, sans-serif; 
                background: #09090b; 
                color: white; 
                display: flex; 
                align-items: center; 
                justify-content: center; 
                height: 100vh; 
                margin: 0;
              }
              .card {
                background: rgba(24,24,27,0.95);
                border: 1px solid rgba(255,255,255,0.1);
                border-radius: 12px;
                padding: 48px;
                max-width: 500px;
                text-align: center;
              }
              .btn {
                background: linear-gradient(135deg, #3b82f633 0%, #3b82f61f 100%);
                border: 1px solid #3b82f680;
                color: white;
                padding: 12px 24px;
                border-radius: 8px;
                text-decoration: none;
                display: inline-block;
                margin-top: 24px;
                cursor: pointer;
              }
              .btn:hover {
                background: linear-gradient(135deg, #3b82f64d 0%, #3b82f62e 100%);
              }
            </style>
          </head>
          <body>
            <div class="card">
              <h1 style="margin: 0 0 16px;">💳 Payment Setup</h1>
              <p style="color: #a1a1aa; line-height: 1.6;">
                You don't have payment information on file yet. 
                You can add payment details when upgrading your subscription plan.
              </p>
              <a href="/admin/subscription" class="btn">← Back to Subscription</a>
            </div>
          </body>
        </html>
      `);
    }
  } catch (error) {
    console.error('Error accessing payment portal:', error);
    res.status(500).send('Failed to access payment portal');
  }
});

// Admin settings (accessible on custom domains)
app.get('/admin/settings', auth.requireAuth, async (req, res) => {
  // Fetch fresh user data from database to avoid session cache issues
  const freshUser = await db.getUserById(req.user.id);
  
  // Map user properties to config (handle both snake_case from DB and camelCase from session)
  const config = {
    siteName: freshUser.site_name || freshUser.username,
    primaryColor: freshUser.primary_color || '#3b82f6',
    logoUrl: freshUser.logo_url || '/nrw-web.png',
    hotChocolateDefaultUrl: freshUser.hot_chocolate_default_url || 'https://square.link/u/NkgDiQCk',
    hotChocolateText: freshUser.hot_chocolate_text || 'Like these pics? Buy me a Hot Chocolate',
    contactEmail: freshUser.contact_email || '',
    socialLinks: freshUser.social_links || { facebook: '', twitter: '', instagram: '', telegram: '', bluesky: '' }
  };
  res.render('admin/settings', { pageTitle: 'Site Settings', config, req: req });
});

// Logo upload - separate multer instance for single file uploads (must be defined before settings route)
const logoUpload = multer({
  storage: multer.diskStorage({
    destination: function (req, file, cb) {
      const logoDir = path.join(__dirname, '..', 'public', 'logos');
      fs.mkdirSync(logoDir, { recursive: true });
      cb(null, logoDir);
    },
    filename: function (req, file, cb) {
      const ext = path.extname(file.originalname);
      const filename = `user-${req.user.id}-logo${ext}`;
      cb(null, filename);
    }
  }),
  fileFilter: function(req, file, cb) {
    const allowedTypes = ['image/png', 'image/jpeg', 'image/jpg', 'image/svg+xml'];
    if (allowedTypes.includes(file.mimetype)) {
      cb(null, true);
    } else {
      cb(new Error('Invalid file type. Only PNG, JPG, and SVG files are allowed.'));
    }
  },
  limits: {
    fileSize: 5 * 1024 * 1024 // 5MB limit for logos
  }
});

// Upload logo (MUST be before /admin/settings POST route for proper routing)
app.post('/admin/settings/logo', auth.requireAuth, logoUpload.single('logo'), async (req, res) => {
  try {
    console.log('[LOGO UPLOAD] Starting logo upload...');
    console.log('[LOGO UPLOAD] User ID:', req.user.id);
    console.log('[LOGO UPLOAD] File received:', req.file ? 'Yes' : 'No');
    
    if (!req.file) {
      console.error('[LOGO UPLOAD] No file in request');
      return res.status(400).send('No logo file uploaded');
    }
    
    console.log('[LOGO UPLOAD] File details:', {
      filename: req.file.filename,
      mimetype: req.file.mimetype,
      size: req.file.size,
      path: req.file.path
    });
    
    const logoUrl = `/logos/${req.file.filename}`;
    
    // If using cloud storage, upload the file
    if (storage.USE_CLOUD_STORAGE) {
      try {
        console.log('[LOGO UPLOAD] Uploading to cloud storage...');
        await storage.uploadFile(req.user.id, req.file.path, `logos/${req.file.filename}`);
        // Delete local temp file after upload
        fs.unlinkSync(req.file.path);
        // Get the public URL from cloud storage
        const cloudLogoUrl = storage.getPublicUrl(req.user.id, null, `logos/${req.file.filename}`, false);
        
        console.log('[LOGO UPLOAD] Cloud logo URL:', cloudLogoUrl);
        
        // Update user's logo URL in database
        await db.updateUserProfile(req.user.id, { logo_url: cloudLogoUrl });
        console.log('[LOGO UPLOAD] Database updated with cloud URL');
      } catch (error) {
        console.error('[LOGO UPLOAD] Error uploading to cloud storage:', error);
        return res.status(500).send('Failed to upload logo to cloud storage');
      }
    } else {
      // Local development - just update the database with the local path
      console.log('[LOGO UPLOAD] Local mode - updating database with:', logoUrl);
      await db.updateUserProfile(req.user.id, { logo_url: logoUrl });
      console.log('[LOGO UPLOAD] Database updated successfully');
    }
    
    console.log('[LOGO UPLOAD] Upload complete! Redirecting...');
    res.redirect('/admin/settings?success=Logo uploaded successfully');
  } catch (error) {
    console.error('[LOGO UPLOAD] Error:', error);
    if (error.message.includes('Invalid file type')) {
      return res.status(400).send(error.message);
    }
    res.status(500).send('Error uploading logo: ' + error.message);
  }
});

// Update settings (must come AFTER /admin/settings/logo route)
app.post('/admin/settings', auth.requireAuth, async (req, res) => {
  try {
    // Ensure req.body exists
    if (!req.body) {
      console.error('[SETTINGS] req.body is undefined');
      return res.status(400).send('Invalid request - no form data received');
    }
    
    const updates = {
      primary_color: String(req.body.primaryColor || '').trim(),
      site_name: String(req.body.siteName || '').trim(),
      hot_chocolate_default_url: String(req.body.hotChocolateDefaultUrl || '').trim(),
      hot_chocolate_text: String(req.body.hotChocolateText || '').trim(),
      contact_email: String(req.body.contactEmail || '').trim(),
      social_links: {
        facebook: String(req.body.facebook || '').trim(),
        twitter: String(req.body.twitter || '').trim(),
        instagram: String(req.body.instagram || '').trim(),
        telegram: String(req.body.telegram || '').trim(),
        bluesky: String(req.body.bluesky || '').trim()
      }
    };
    
    await db.updateUserProfile(req.user.id, updates);
    
    // Check if this is an AJAX request (autosave)
    const isAjax = req.get('Content-Type') === 'application/x-www-form-urlencoded' && 
                   !req.get('Referer')?.includes('?success=');
    
    if (isAjax) {
      // Return JSON for autosave requests
      return res.json({ success: true, message: 'Settings saved' });
    } else {
      // Redirect for traditional form submissions
      res.redirect('/admin/settings?success=Settings saved successfully');
    }
  } catch (error) {
    console.error('[SETTINGS] Error updating settings:', error);
    
    // Check if AJAX request
    const isAjax = req.get('Content-Type') === 'application/x-www-form-urlencoded';
    
    if (isAjax) {
      return res.status(500).json({ success: false, message: error.message });
    } else {
      res.status(500).send('Error updating settings: ' + error.message);
    }
  }
});

// Create album
app.post('/admin/albums', auth.requireAuth, async (req, res) => {
  const title = String(req.body.title || '').trim();
  
  if (!validateTitle(title)) {
    return res.status(400).send('Invalid album title (max 200 characters)');
  }
  
  // Check if user has reached album limit (for free tier)
  if (await db.hasReachedAlbumLimit(req.user.id)) {
    return res.status(403).send('You have reached your album limit. Please upgrade to create more albums.');
  }
  
  const code = nanoShort();
  const id = await db.createAlbum(req.user.id, title, code);
  return res.redirect(`/admin/albums/${id}`);
});

// View album
app.get('/admin/albums/:id', auth.requireAuth, async (req, res) => {
  const id = Number(req.params.id);
  const album = await db.getAlbumById(id, req.user.id);
  if (!album) return res.redirect('/admin');
  
  const photos = await db.getPhotosByAlbum(id);
  const groups = await db.listGroups(req.user.id);
  const albumGroup = await db.getGroupForAlbum(id);
  
  res.render('admin/album', {
    pageTitle: album.title,
    album,
    photos,
    groups,
    albumGroup,
    galleryUrl: tenant.getGalleryUrl(req.user, req),
    siteConfig: {
      siteName: req.user.siteName || req.user.site_name || req.user.username,
      primaryColor: req.user.primaryColor || req.user.primary_color || '#3b82f6',
      primaryColorRgb: hexToRgb(req.user.primaryColor || req.user.primary_color || '#3b82f6'),
      logoUrl: req.user.logoUrl || req.user.logo_url || '/nrw-web.png',
      hotChocolateDefaultUrl: req.user.hotChocolateDefaultUrl || req.user.hot_chocolate_default_url || 'https://square.link/u/NkgDiQCk',
      hotChocolateText: req.user.hotChocolateText || req.user.hot_chocolate_text || 'Like these pics? Buy me a Hot Chocolate',
      contactEmail: req.user.contactEmail || req.user.contact_email || '',
      socialLinks: req.user.socialLinks || req.user.social_links || {},
      ageVerificationEnabled: req.user.requireAgeVerification || req.user.require_age_verification || false
    },
    req: req
  });
});

// Upload photos
app.post('/admin/albums/:id/upload', auth.requireAuth, uploadLimiter, (req, res, next) => {
  upload.array('photos', 500)(req, res, (err) => {
    if (err) {
      console.error('Multer error:', err);
      return res.status(400).send(`Upload error: ${err.message}`);
    }
    next();
  });
}, async (req, res) => {
  try {
    const id = Number(req.params.id);
    const album = await db.getAlbumById(id, req.user.id);
    if (!album) return res.status(404).send('Album not found');
    
    // Check if files were uploaded
    if (!req.files || req.files.length === 0) {
      return res.status(400).send('No files uploaded. Please select image files.');
    }
    
    // Check limits before uploading
    const totalSize = req.files.reduce((sum, file) => sum + file.size, 0);
    
    console.log(`[Upload Check] User: ${req.user.username}, Files: ${req.files.length}, Total Size: ${(totalSize / 1024 / 1024).toFixed(2)}MB`);
    console.log(`[Upload Check] Current: ${req.user.current_photo_count}/${req.user.max_photos} photos, ${(req.user.current_storage_bytes / 1024 / 1024 / 1024).toFixed(2)}GB/${(req.user.max_storage_bytes / 1024 / 1024 / 1024).toFixed(2)}GB storage`);
    
    if (await db.hasReachedPhotoLimit(req.user.id)) {
      console.log('[Upload Check] Photo limit reached!');
      // Delete uploaded files
      for (const file of req.files || []) {
        fs.unlinkSync(file.path);
      }
      return res.status(403).send('Photo limit reached. Please upgrade your plan.');
    }
    
    const storageLimit = await db.hasReachedStorageLimit(req.user.id, totalSize);
    console.log(`[Upload Check] Storage limit check: ${storageLimit} (current: ${req.user.current_storage_bytes}, additional: ${totalSize}, max: ${req.user.max_storage_bytes})`);
    
    if (storageLimit) {
      console.log('[Upload Check] Storage limit reached!');
      // Delete uploaded files
      for (const file of req.files || []) {
        fs.unlinkSync(file.path);
      }
      return res.status(403).send('Storage limit reached. Please upgrade your plan.');
    }
    
    // Add photos to DB and upload to Cloud Storage if enabled
    console.log(`[Upload] Processing ${req.files.length} files...`);
    
    for (const file of req.files || []) {
      console.log(`[Upload] Processing file: ${file.originalname} (${(file.size / 1024 / 1024).toFixed(2)}MB)`);
      
      if (storage.USE_CLOUD_STORAGE) {
        console.log(`[Upload] Uploading to cloud storage...`);
        await storage.uploadFile(req.user.id, file.path, `${id}/${file.filename}`);
        fs.unlinkSync(file.path);
      }
      
      console.log(`[Upload] Adding to database...`);
      await db.addPhoto(id, req.user.id, file.filename, file.originalname, file.mimetype, file.size);
      
      console.log(`[Upload] Generating thumbnail...`);
      generateThumbnail(req.user.id, id, file.filename).catch(err =>
        console.error(`Failed to generate thumbnail for ${file.filename}:`, err)
      );
    }
    
    // Update user's usage statistics
    console.log(`[Upload] Updating user usage statistics...`);
    await db.updateUserUsage(req.user.id, req.files.length, totalSize);
    
    console.log(`[Upload] Complete!`);
    console.log(`[Upload] req.xhr: ${req.xhr}, Accept header: ${req.headers.accept}`);
    
    // Always return JSON for successful uploads (XHR requests expect it)
    // The client-side JavaScript will handle the page reload
    console.log(`[Upload] Sending JSON response...`);
    return res.status(200).json({ 
      success: true, 
      uploaded: req.files.length,
      message: 'Upload successful' 
    });
  } catch (error) {
    console.error('Upload error:', error);
    // Clean up any uploaded files on error
    if (req.files) {
      for (const file of req.files) {
        try {
          if (fs.existsSync(file.path)) {
            fs.unlinkSync(file.path);
          }
        } catch (cleanupErr) {
          console.error('Error cleaning up file:', cleanupErr);
        }
      }
    }
    return res.status(500).send(`Upload failed: ${error.message}`);
  }
});

// Delete photo
app.post('/admin/albums/:albumId/photos/:photoId/delete', auth.requireAuth, async (req, res) => {
  try {
    const photoId = Number(req.params.photoId);
    const photo = await db.getPhotoById(photoId);
    
    if (!photo) {
      return res.status(404).json({ success: false, message: 'Photo not found' });
    }
    
    // Verify the photo's album belongs to this user
    const album = await db.getAlbumById(photo.album_id, req.user.id);
    if (!album) {
      return res.status(403).json({ success: false, message: 'Access denied' });
    }
    
    // Delete the photo files
    try {
      // Delete full-size image
      const fullPath = path.join(__dirname, '..', 'uploads', String(req.user.id), String(photo.album_id), photo.filename);
      if (fs.existsSync(fullPath)) {
        fs.unlinkSync(fullPath);
      }
      
      // Delete thumbnail
      const thumbPath = path.join(__dirname, '..', 'uploads', String(req.user.id), String(photo.album_id), 'thumbs', photo.filename);
      if (fs.existsSync(thumbPath)) {
        fs.unlinkSync(thumbPath);
      }
    } catch (fileErr) {
      console.error('Error deleting photo files:', fileErr);
      // Continue with database deletion even if file deletion fails
    }
    
    // Delete from database
    await db.deletePhoto(photoId, req.user.id);
    
    // Update user usage statistics
    await db.updateUserUsage(req.user.id, -1, -photo.size_bytes);
    
    res.json({ success: true });
  } catch (error) {
    console.error('Delete photo error:', error);
    res.status(500).json({ success: false, message: error.message });
  }
});

// Reorder photos
app.post('/admin/albums/:id/photos/reorder', auth.requireAuth, async (req, res) => {
  try {
    const albumId = Number(req.params.id);
    const photoIds = req.body.photoIds;
    
    // Verify album belongs to user
    const album = await db.getAlbumById(albumId, req.user.id);
    if (!album) {
      return res.status(403).json({ success: false, message: 'Access denied' });
    }
    
    await db.updatePhotoSortOrder(photoIds);
    res.json({ success: true });
  } catch (error) {
    console.error('Reorder photos error:', error);
    res.status(500).json({ success: false, message: error.message });
  }
});

// All other admin routes remain similar but with user_id filtering
// (Album management, photo management, group management, etc.)
// I'll include key ones for completeness

app.post('/admin/albums/:id/rename', auth.requireAuth, async (req, res) => {
  const id = Number(req.params.id);
  const title = String(req.body.title || '').trim();
  
  if (!validateTitle(title)) {
    return res.status(400).send('Invalid album title');
  }
  
  await db.updateAlbumTitle(id, req.user.id, title);
  res.redirect(`/admin/albums/${id}`);
});

app.post('/admin/albums/:id/password', auth.requireAuth, async (req, res) => {
  const id = Number(req.params.id);
  const password = String(req.body.password || '').trim();
  const removePassword = req.body.remove === '1';
  
  try {
    if (removePassword) {
      await db.updateAlbumPassword(id, req.user.id, null);
    } else if (password) {
      await db.updateAlbumPassword(id, req.user.id, password);
    }
    res.redirect(`/admin/albums/${id}`);
  } catch (error) {
    console.error('Error updating album password:', error);
    res.status(500).send('Failed to update password');
  }
});

app.post('/admin/albums/:id/delete', auth.requireAuth, async (req, res) => {
  const id = Number(req.params.id);
  await db.deleteAlbum(id, req.user.id);
  res.redirect('/admin');
});

app.post('/admin/groups', auth.requireAuth, async (req, res) => {
  const name = String(req.body.name || '').trim();
  const description = String(req.body.description || '').trim() || null;
  if (!name) return res.redirect('/admin');
  
  const code = nanoShort();
  const id = await db.createGroup(req.user.id, name, code, description);
  return res.redirect(`/admin/groups/${id}`);
});

// View group
app.get('/admin/groups/:id', auth.requireAuth, async (req, res) => {
  const id = Number(req.params.id);
  const group = await db.getGroupById(id, req.user.id);
  if (!group) return res.redirect('/admin');
  
  const albums = await db.getAlbumsByGroup(id);
  
  res.render('admin/group', {
    pageTitle: group.name,
    group,
    albums,
    galleryUrl: tenant.getGalleryUrl(req.user, req),
    siteConfig: {
      siteName: req.user.siteName || req.user.site_name || req.user.username,
      primaryColor: req.user.primaryColor || req.user.primary_color || '#3b82f6',
      primaryColorRgb: hexToRgb(req.user.primaryColor || req.user.primary_color || '#3b82f6'),
      logoUrl: req.user.logoUrl || req.user.logo_url || '/nrw-web.png',
      hotChocolateDefaultUrl: req.user.hotChocolateDefaultUrl || req.user.hot_chocolate_default_url || 'https://square.link/u/NkgDiQCk',
      hotChocolateText: req.user.hotChocolateText || req.user.hot_chocolate_text || 'Like these pics? Buy me a Hot Chocolate',
      contactEmail: req.user.contactEmail || req.user.contact_email || '',
      socialLinks: req.user.socialLinks || req.user.social_links || {},
      ageVerificationEnabled: req.user.requireAgeVerification || req.user.require_age_verification || false
    },
    req: req
  });
});

// Delete group (confirmation page)
app.get('/admin/groups/:id/delete', auth.requireAuth, async (req, res) => {
  const id = Number(req.params.id);
  const group = await db.getGroupById(id, req.user.id);
  if (!group) return res.redirect('/admin');
  
  const albums = await db.getAlbumsByGroup(id);
  
  res.render('admin/delete-group', {
    pageTitle: `Delete ${group.name}`,
    group,
    albums,
    siteConfig: {
      siteName: req.user.siteName || req.user.site_name || req.user.username,
      primaryColor: req.user.primaryColor || req.user.primary_color || '#3b82f6',
      primaryColorRgb: hexToRgb(req.user.primaryColor || req.user.primary_color || '#3b82f6'),
      logoUrl: req.user.logoUrl || req.user.logo_url || '/nrw-web.png',
      hotChocolateDefaultUrl: req.user.hotChocolateDefaultUrl || req.user.hot_chocolate_default_url || 'https://square.link/u/NkgDiQCk',
      hotChocolateText: req.user.hotChocolateText || req.user.hot_chocolate_text || 'Like these pics? Buy me a Hot Chocolate',
      contactEmail: req.user.contactEmail || req.user.contact_email || '',
      socialLinks: req.user.socialLinks || req.user.social_links || {},
      ageVerificationEnabled: req.user.requireAgeVerification || req.user.require_age_verification || false
    },
    req: req
  });
});

// Delete group (action)
app.post('/admin/groups/:id/delete', auth.requireAuth, async (req, res) => {
  const id = Number(req.params.id);
  const deleteAlbums = req.body.delete_albums === '1';
  
  await db.deleteGroup(id, req.user.id, deleteAlbums);
  res.redirect('/admin');
});

// ============================================================================
// SUBSCRIPTION & BILLING ROUTES
// ============================================================================

const stripeWebhook = require('./stripe-webhook');

// Subscription management page
app.get('/admin/subscription', auth.requireAuth, async (req, res) => {
  try {
    const user = await db.getUserById(req.user.id);
    
    if (!user) {
      return res.status(404).send('User not found');
    }
    
    // Get usage statistics
    const usage = await db.getUserUsage(user.id);
    
    // Get album count
    const albums = await db.listAlbumsByUser(user.id);
    const albumCount = albums.length;
    
    res.render('admin/subscription', {
      pageTitle: 'Subscription',
      user,
      usage,
      albumCount,
      siteConfig: {
        siteName: user.siteName || user.site_name || user.username,
        primaryColor: user.primaryColor || user.primary_color || '#3b82f6',
        primaryColorRgb: hexToRgb(user.primaryColor || user.primary_color || '#3b82f6'),
        logoUrl: user.logoUrl || user.logo_url || '/nrw-web.png',
        hotChocolateDefaultUrl: user.hotChocolateDefaultUrl || user.hot_chocolate_default_url || 'https://square.link/u/NkgDiQCk',
        hotChocolateText: user.hotChocolateText || user.hot_chocolate_text || 'Like these pics? Buy me a Hot Chocolate',
        contactEmail: user.contactEmail || user.contact_email || '',
        socialLinks: user.socialLinks || user.social_links || {},
        ageVerificationEnabled: user.requireAgeVerification || user.require_age_verification || false
      },
      req: req
    });
  } catch (error) {
    console.error('[SUBSCRIPTION] Error loading subscription page:', error);
    res.status(500).send('Error loading subscription');
  }
});

// Change subscription tier
app.post('/admin/subscription/change', auth.requireAuth, async (req, res) => {
  try {
    const user = await db.getUserById(req.user.id);
    const newTier = req.body.tier;
    
    if (!user) {
      return res.status(404).json({ error: 'User not found' });
    }
    
    // Validate tier
    const validTiers = ['free', 'starter', 'professional', 'business'];
    if (!validTiers.includes(newTier)) {
      return res.status(400).json({ error: 'Invalid tier' });
    }
    
    // Handle downgrade to free
    if (newTier === 'free') {
      if (user.stripe_subscription_id) {
        // Cancel Stripe subscription
        await stripeService.cancelSubscription(user.stripe_subscription_id);
      }
      
      // Update user to free tier
      await db.updateUser(user.id, {
        subscriptionTier: 'free',
        subscriptionStatus: 'active',
        stripeSubscriptionId: null,
      });
      
      return res.redirect('/admin/subscription?success=downgraded');
    }
    
    // Handle upgrade/downgrade with Stripe
    if (!stripeService.isConfigured()) {
      console.warn('[STRIPE] Stripe not configured, performing manual tier change');
      await db.updateSubscriptionTier(user.id, newTier);
      return res.redirect('/admin/subscription?success=changed');
    }
    
    // If user doesn't have a Stripe customer ID, create one
    if (!user.stripe_customer_id) {
      const customerId = await stripeService.createCustomer(
        user.email,
        user.username,
        user.full_name
      );
      
      await db.updateUser(user.id, {
        stripeCustomerId: customerId,
      });
      
      user.stripe_customer_id = customerId;
    }
    
    // If user has an existing subscription, update it
    if (user.stripe_subscription_id) {
      await stripeService.updateSubscription(user.stripe_subscription_id, newTier);
      
      // Update user tier (webhook will update other fields)
      await db.updateUser(user.id, {
        subscriptionTier: newTier,
      });
      
      return res.redirect('/admin/subscription?success=changed');
    }
    
    // Create new subscription with checkout
    const protocol = req.secure || req.headers['x-forwarded-proto'] === 'https' ? 'https' : 'http';
    const baseUrl = `${protocol}://${req.get('host')}`;
    const successUrl = `${baseUrl}/admin/subscription?success=subscribed`;
    const cancelUrl = `${baseUrl}/admin/subscription?canceled=true`;
    
    const checkoutUrl = await stripeService.createCheckoutSession(
      user.stripe_customer_id,
      newTier,
      successUrl,
      cancelUrl
    );
    
    return res.redirect(checkoutUrl);
  } catch (error) {
    console.error('[SUBSCRIPTION] Error changing subscription:', error);
    res.redirect('/admin/subscription?error=change_failed');
  }
});

// Cancel subscription
app.post('/admin/subscription/cancel', auth.requireAuth, async (req, res) => {
  try {
    const user = await db.getUserById(req.user.id);
    
    if (!user) {
      return res.status(404).json({ error: 'User not found' });
    }
    
    if (!user.stripe_subscription_id) {
      return res.redirect('/admin/subscription?error=no_subscription');
    }
    
    // Cancel subscription at period end
    const result = await stripeService.cancelSubscription(user.stripe_subscription_id);
    
    // Update user status
    await db.updateUser(user.id, {
      subscriptionStatus: 'canceled',
      subscriptionEndsAt: result.cancelAt,
    });
    
    res.redirect('/admin/subscription?success=canceled');
  } catch (error) {
    console.error('[SUBSCRIPTION] Error canceling subscription:', error);
    res.redirect('/admin/subscription?error=cancel_failed');
  }
});

// Billing portal
app.post('/admin/subscription/payment-portal', auth.requireAuth, async (req, res) => {
  try {
    const user = await db.getUserById(req.user.id);
    
    if (!user) {
      return res.status(404).json({ error: 'User not found' });
    }
    
    if (!user.stripe_customer_id) {
      return res.redirect('/admin/subscription?error=no_customer');
    }
    
    // Create billing portal session
    const protocol = req.secure || req.headers['x-forwarded-proto'] === 'https' ? 'https' : 'http';
    const returnUrl = `${protocol}://${req.get('host')}/admin/subscription`;
    
    const portalUrl = await stripeService.createBillingPortalSession(
      user.stripe_customer_id,
      returnUrl
    );
    
    res.redirect(portalUrl);
  } catch (error) {
    console.error('[SUBSCRIPTION] Error creating portal session:', error);
    res.redirect('/admin/subscription?error=portal_failed');
  }
});

// Stripe webhook endpoint (MUST be before express.json middleware for raw body)
// This is placed here to catch it early - the webhook needs raw body
app.post('/webhooks/stripe', express.raw({ type: 'application/json' }), async (req, res) => {
  const signature = req.headers['stripe-signature'];
  const webhookSecret = process.env.STRIPE_WEBHOOK_SECRET;
  
  if (!webhookSecret) {
    console.error('[STRIPE WEBHOOK] Webhook secret not configured');
    return res.status(500).json({ error: 'Webhook secret not configured' });
  }
  
  try {
    // Verify webhook signature
    const event = stripeService.constructWebhookEvent(
      req.body,
      signature,
      webhookSecret
    );
    
    // Handle the event
    await stripeWebhook.handleWebhook(db, event);
    
    res.json({ received: true });
  } catch (error) {
    console.error('[STRIPE WEBHOOK] Error processing webhook:', error);
    res.status(400).json({ error: `Webhook Error: ${error.message}` });
  }
});

// ============================================================================
// SUPERUSER PORTAL ROUTES
// ============================================================================

// Superuser dashboard
app.get('/superuser', requireAppSubdomain, auth.requireAuth, superuser.requireSuperuser, async (req, res) => {
  try {
    console.log('Loading superuser dashboard...');
    
    console.log('Fetching user statistics...');
    const stats = await superuser.getUserStatistics();
    console.log('Stats:', stats);
    
    console.log('Fetching revenue statistics...');
    const revenue = await superuser.getRevenueStatistics();
    console.log('Revenue:', revenue);
    
    console.log('Fetching recent actions...');
    const recentActions = await superuser.getRecentAdminActions(20);
    console.log('Recent actions count:', recentActions.length);
    
    res.render('superuser/dashboard', {
      pageTitle: 'Superuser Dashboard',
      stats,
      revenue,
      recentActions,
      req: req
    });
  } catch (error) {
    console.error('Error loading superuser dashboard:', error);
    console.error('Error stack:', error.stack);
    res.status(500).send(`Error loading dashboard: ${error.message}`);
  }
});

// User management
app.get('/superuser/users', requireAppSubdomain, auth.requireAuth, superuser.requireSuperuser, async (req, res) => {
  try {
    const page = parseInt(req.query.page, 10) || 1;
    const search = req.query.search || null;
    const tier = req.query.tier || null;
    const status = req.query.status || null;
    const sortBy = req.query.sortBy || 'created_at';
    
    const data = await superuser.getAllUsers({
      page,
      limit: 50,
      search,
      tier,
      status,
      sortBy
    });
    
    res.render('superuser/users', {
      pageTitle: 'User Management',
      data,
      filters: { search, tier, status, sortBy },
      req: req
    });
  } catch (error) {
    console.error('Error loading users:', error);
    res.status(500).send('Error loading users');
  }
});

// New user form
app.get('/superuser/users/new', requireAppSubdomain, auth.requireAuth, superuser.requireSuperuser, async (req, res) => {
  try {
    const tiers = await superuser.getAllTiers();
    
    res.render('superuser/user-new', {
      pageTitle: 'Create New User',
      tiers,
      baseDomain: tenant.BASE_DOMAIN,
      error: null,
      formData: {},
      req: req
    });
  } catch (error) {
    console.error('Error loading new user form:', error);
    res.status(500).send('Error loading form');
  }
});

// Create new user
app.post('/superuser/users/create', requireAppSubdomain, auth.requireAuth, superuser.requireSuperuser, async (req, res) => {
  try {
    const { username, email, password, full_name, subscription_tier, subscription_status } = req.body;
    
    const userData = {
      username: username.toLowerCase().trim(),
      email: email.toLowerCase().trim(),
      password,
      full_name: full_name || null,
      subscription_tier: subscription_tier || 'starter',
      subscription_status: subscription_status || 'trialing'
    };
    
    const newUser = await superuser.createUserManually(req.user.id, userData, req);
    
    res.redirect(`/superuser/users/${newUser.id}`);
  } catch (error) {
    console.error('Error creating user:', error);
    
    // Re-render form with error
    const tiers = await superuser.getAllTiers();
    res.status(400).render('superuser/user-new', {
      pageTitle: 'Create New User',
      tiers,
      baseDomain: tenant.BASE_DOMAIN,
      error: error.message,
      formData: req.body,
      req: req
    });
  }
});

// Edit user
app.get('/superuser/users/:id', requireAppSubdomain, auth.requireAuth, superuser.requireSuperuser, async (req, res) => {
  try {
    const userId = parseInt(req.params.id, 10);
    const user = await db.getUserById(userId);
    
    if (!user) {
      return res.status(404).send('User not found');
    }
    
    const tiers = await superuser.getAllTiers();
    
    res.render('superuser/user-edit', {
      pageTitle: `Edit User: ${user.username}`,
      user,
      tiers,
      baseDomain: tenant.BASE_DOMAIN,
      req: req
    });
  } catch (error) {
    console.error('Error loading user:', error);
    res.status(500).send('Error loading user');
  }
});

// Update user tier
app.post('/superuser/users/:id/update-tier', requireAppSubdomain, auth.requireAuth, superuser.requireSuperuser, async (req, res) => {
  try {
    const userId = parseInt(req.params.id, 10);
    const newTier = req.body.tier;
    
    await superuser.updateUserTier(req.user.id, userId, newTier, req);
    
    res.redirect(`/superuser/users/${userId}`);
  } catch (error) {
    console.error('Error updating user tier:', error);
    res.status(500).send('Error updating tier: ' + error.message);
  }
});

// Update user limits
app.post('/superuser/users/:id/update-limits', requireAppSubdomain, auth.requireAuth, superuser.requireSuperuser, async (req, res) => {
  try {
    const userId = parseInt(req.params.id, 10);
    const maxPhotos = parseInt(req.body.maxPhotos, 10);
    const maxStorageGB = parseFloat(req.body.maxStorageGB);
    const maxStorageBytes = Math.round(maxStorageGB * 1024 * 1024 * 1024);
    
    await superuser.updateUserLimits(req.user.id, userId, maxPhotos, maxStorageBytes, req);
    
    res.redirect(`/superuser/users/${userId}`);
  } catch (error) {
    console.error('Error updating user limits:', error);
    res.status(500).send('Error updating limits: ' + error.message);
  }
});

// Suspend user
app.post('/superuser/users/:id/suspend', requireAppSubdomain, auth.requireAuth, superuser.requireSuperuser, async (req, res) => {
  try {
    const userId = parseInt(req.params.id, 10);
    const reason = req.body.reason || null;
    
    await superuser.suspendUser(req.user.id, userId, reason, req);
    
    res.redirect(`/superuser/users/${userId}`);
  } catch (error) {
    console.error('Error suspending user:', error);
    res.status(500).send('Error suspending user: ' + error.message);
  }
});

// Activate user
app.post('/superuser/users/:id/activate', requireAppSubdomain, auth.requireAuth, superuser.requireSuperuser, async (req, res) => {
  try {
    const userId = parseInt(req.params.id, 10);
    
    await superuser.activateUser(req.user.id, userId, req);
    
    res.redirect(`/superuser/users/${userId}`);
  } catch (error) {
    console.error('Error activating user:', error);
    res.status(500).send('Error activating user: ' + error.message);
  }
});

// Delete user
app.post('/superuser/users/:id/delete', requireAppSubdomain, auth.requireAuth, superuser.requireSuperuser, async (req, res) => {
  try {
    const userId = parseInt(req.params.id, 10);
    const confirmUsername = req.body.confirmUsername;
    const user = await db.getUserById(userId);
    
    if (!user) {
      return res.status(404).send('User not found');
    }
    
    if (confirmUsername !== user.username) {
      return res.status(400).send('Username confirmation does not match');
    }
    
    await superuser.deleteUser(req.user.id, userId, req);
    
    res.redirect('/superuser/users');
  } catch (error) {
    console.error('Error deleting user:', error);
    res.status(500).send('Error deleting user: ' + error.message);
  }
});

// Make user superuser
app.post('/superuser/users/:id/make-superuser', requireAppSubdomain, auth.requireAuth, superuser.requireSuperuser, async (req, res) => {
  try {
    const userId = parseInt(req.params.id, 10);
    
    await superuser.makeSuperuser(req.user.id, userId, req);
    
    res.redirect(`/superuser/users/${userId}`);
  } catch (error) {
    console.error('Error making superuser:', error);
    res.status(500).send('Error making superuser: ' + error.message);
  }
});

// Remove superuser status
app.post('/superuser/users/:id/remove-superuser', requireAppSubdomain, auth.requireAuth, superuser.requireSuperuser, async (req, res) => {
  try {
    const userId = parseInt(req.params.id, 10);
    
    await superuser.removeSuperuser(req.user.id, userId, req);
    
    res.redirect(`/superuser/users/${userId}`);
  } catch (error) {
    console.error('Error removing superuser:', error);
    res.status(500).send('Error removing superuser: ' + error.message);
  }
});

// Tier management
app.get('/superuser/tiers', requireAppSubdomain, auth.requireAuth, superuser.requireSuperuser, async (req, res) => {
  try {
    const tiers = await superuser.getAllTiers();
    
    res.render('superuser/tiers', {
      pageTitle: 'Subscription Tiers',
      tiers,
      req: req
    });
  } catch (error) {
    console.error('Error loading tiers:', error);
    res.status(500).send('Error loading tiers');
  }
});

// New tier form
app.get('/superuser/tiers/new', requireAppSubdomain, auth.requireAuth, superuser.requireSuperuser, async (req, res) => {
  res.render('superuser/tier-form', {
    pageTitle: 'Create New Tier',
    tier: null,
    req: req
  });
});

// Edit tier form
app.get('/superuser/tiers/:id/edit', requireAppSubdomain, auth.requireAuth, superuser.requireSuperuser, async (req, res) => {
  try {
    const tierId = parseInt(req.params.id, 10);
    const tier = await superuser.getTierById(tierId);
    
    if (!tier) {
      return res.status(404).send('Tier not found');
    }
    
    res.render('superuser/tier-form', {
      pageTitle: `Edit Tier: ${tier.tier_name}`,
      tier,
      req: req
    });
  } catch (error) {
    console.error('Error loading tier:', error);
    res.status(500).send('Error loading tier');
  }
});

// Create tier
app.post('/superuser/tiers/create', requireAppSubdomain, auth.requireAuth, superuser.requireSuperuser, async (req, res) => {
  try {
    const pricePounds = parseFloat(req.body.price_pounds);
    const maxStorageGB = parseFloat(req.body.max_storage_gb);
    
    const tierData = {
      tier_key: req.body.tier_key.toLowerCase().trim(),
      tier_name: req.body.tier_name.trim(),
      price_pence: Math.round(pricePounds * 100),
      max_photos: parseInt(req.body.max_photos, 10),
      max_storage_bytes: Math.round(maxStorageGB * 1024 * 1024 * 1024),
      allows_custom_domain: req.body.allows_custom_domain === 'on',
      allows_multiple_domains: req.body.allows_multiple_domains === 'on',
      description: req.body.description || null,
      features: Array.isArray(req.body.features) ? req.body.features.filter(f => f.trim()) : []
    };
    
    await superuser.createTier(req.user.id, tierData, req);
    
    res.redirect('/superuser/tiers');
  } catch (error) {
    console.error('Error creating tier:', error);
    res.status(500).send('Error creating tier: ' + error.message);
  }
});

// Update tier
app.post('/superuser/tiers/:id/update', requireAppSubdomain, auth.requireAuth, superuser.requireSuperuser, async (req, res) => {
  try {
    const tierId = parseInt(req.params.id, 10);
    const pricePounds = parseFloat(req.body.price_pounds);
    const maxStorageGB = parseFloat(req.body.max_storage_gb);
    
    const tierData = {
      tier_name: req.body.tier_name.trim(),
      price_pence: Math.round(pricePounds * 100),
      max_photos: parseInt(req.body.max_photos, 10),
      max_storage_bytes: Math.round(maxStorageGB * 1024 * 1024 * 1024),
      allows_custom_domain: req.body.allows_custom_domain === 'on',
      allows_multiple_domains: req.body.allows_multiple_domains === 'on',
      display_order: parseInt(req.body.display_order, 10) || 0,
      is_active: req.body.is_active === 'on',
      description: req.body.description || null,
      features: Array.isArray(req.body.features) ? req.body.features.filter(f => f.trim()) : []
    };
    
    await superuser.updateTier(req.user.id, tierId, tierData, req);
    
    res.redirect('/superuser/tiers');
  } catch (error) {
    console.error('Error updating tier:', error);
    res.status(500).send('Error updating tier: ' + error.message);
  }
});

// Deactivate tier
app.post('/superuser/tiers/:id/deactivate', requireAppSubdomain, auth.requireAuth, superuser.requireSuperuser, async (req, res) => {
  try {
    const tierId = parseInt(req.params.id, 10);
    
    await superuser.deactivateTier(req.user.id, tierId, req);
    
    res.redirect('/superuser/tiers');
  } catch (error) {
    console.error('Error deactivating tier:', error);
    res.status(500).send('Error deactivating tier: ' + error.message);
  }
});

// Activate tier
app.post('/superuser/tiers/:id/activate', requireAppSubdomain, auth.requireAuth, superuser.requireSuperuser, async (req, res) => {
  try {
    const tierId = parseInt(req.params.id, 10);
    
    await superuser.updateTier(req.user.id, tierId, { is_active: true }, req);
    
    res.redirect('/superuser/tiers');
  } catch (error) {
    console.error('Error activating tier:', error);
    res.status(500).send('Error activating tier: ' + error.message);
  }
});

// Audit log
app.get('/superuser/audit-log', requireAppSubdomain, auth.requireAuth, superuser.requireSuperuser, async (req, res) => {
  try {
    const logs = await superuser.getRecentAdminActions(100);
    
    res.render('superuser/audit-log', {
      pageTitle: 'Admin Audit Log',
      logs,
      req: req
    });
  } catch (error) {
    console.error('Error loading audit log:', error);
    res.status(500).send('Error loading audit log');
  }
});

// 404 fallback
app.use((req, res) => {
  res.status(404).render('not-found', { pageTitle: 'Not Found', req: req });
});

// Run migrations on startup
async function runMigrations() {
  try {
    await db.pool.query(`
      ALTER TABLE albums ADD COLUMN IF NOT EXISTS password_hash TEXT;
      CREATE INDEX IF NOT EXISTS idx_albums_password_hash ON albums(password_hash) WHERE password_hash IS NOT NULL;
    `);
    console.log('✓ Database migrations applied');
  } catch (error) {
    console.error('Migration error (may already be applied):', error.message);
  }
}

// Start server
runMigrations().then(() => {
  app.listen(PORT, () => {
    console.log(`Vybe Photo multi-tenant server running on http://localhost:${PORT}`);
    console.log(`Base domain: ${tenant.BASE_DOMAIN}`);
  });
});

