<?php

namespace App\Services\AI;

use Google\Auth\ApplicationDefaultCredentials;
use Google\Auth\Credentials\CredentialsLoader;
use Google\Auth\Credentials\ServiceAccountCredentials;
use GuzzleHttp\Client as HttpClient;
use GuzzleHttp\Exception\GuzzleException;

class VertexClient
{
    private string $project;
    private string $location;
    private HttpClient $http;
    private ?string $apiKey = null;
    private ?string $cachedToken = null;
    private int $tokenExpiry = 0;

    public function __construct()
    {
        $this->project = (string) (config('vertex.project') ?? '');
        $this->location = (string) (config('vertex.location') ?? 'us-central1');
        $this->apiKey = config('vertex.api_key');
        
        // Use location-specific endpoint for Vertex AI with optimized settings
        $this->http = new HttpClient([
            'base_uri' => sprintf('https://%s-aiplatform.googleapis.com', $this->location),
            'timeout' => config('vertex.timeout', 30), // Reduced timeout for faster failure detection
            'connect_timeout' => 5, // Fast connection timeout
            'http_errors' => false, // Don't throw exceptions for HTTP errors
            'verify' => false, // Skip SSL verification for faster connections (dev only)
            'headers' => [
                'User-Agent' => 'ExpenseWise/1.0',
                'Accept' => 'application/json',
                'Accept-Encoding' => 'gzip, deflate',
            ],
        ]);
    }

    private function getAccessToken(): ?string
    {
        // Return cached token if still valid
        if ($this->cachedToken && time() < $this->tokenExpiry) {
            return $this->cachedToken;
        }
        
        $scopes = ['https://www.googleapis.com/auth/cloud-platform'];
        try {
            if ($json = env('GOOGLE_APPLICATION_CREDENTIALS_JSON')) {
                $creds = ServiceAccountCredentials::fromJson(json_decode($json, true));
                $creds->setScopes($scopes);
                $token = $creds->fetchAuthToken();
                $this->cachedToken = $token['access_token'] ?? null;
                $this->tokenExpiry = time() + 3000; // Cache for 50 minutes (tokens last 1 hour)
                return $this->cachedToken;
            }
            $creds = ApplicationDefaultCredentials::getCredentials($scopes);
            $token = $creds->fetchAuthToken();
            $this->cachedToken = $token['access_token'] ?? null;
            $this->tokenExpiry = time() + 3000; // Cache for 50 minutes
            return $this->cachedToken;
        } catch (\Throwable $e) {
            \Log::error('Vertex AI: Failed to get access token', ['error' => $e->getMessage()]);
            return null;
        }
    }

    public function generate(string $model, string $prompt, ?string $system = null, array $params = []): array
    {
        if (!$this->project || !$this->location) {
            return ['text' => null, 'json' => null];
        }

        // Get OAuth2 access token
        $accessToken = $this->getAccessToken();
        if (!$accessToken) {
            \Log::error('Vertex AI: No access token available');
            return ['text' => null, 'json' => null];
        }

        // Reuse existing HTTP client with optimized settings
        $endpoint = sprintf('/v1/projects/%s/locations/%s/publishers/google/models/%s:generateContent', $this->project, $this->location, $model);

        // Parse the prompt to check if it contains image data
        $promptData = json_decode($prompt, true);
        $parts = [['text' => $prompt]];

        if (is_array($promptData) && isset($promptData['image'])) {
            // Handle multimodal input with image; allow caller to specify correct mime
            $mime = $promptData['image_mime'] ?? 'image/jpeg';
            $parts = [
                ['text' => $promptData['instructions'] ?? $prompt],
                ['inline_data' => [
                    'mime_type' => $mime,
                    'data' => $promptData['image']
                ]]
            ];
        }

        $body = [
            'contents' => [
                [
                    'role' => 'user',
                    'parts' => $parts,
                ],
            ],
            'generationConfig' => array_merge([
                'temperature' => (float) config('vertex.temperature', 0.2),
                'topP' => (float) config('vertex.top_p', 0.95),
                'topK' => (int) config('vertex.top_k', 40),
            ], $params),
        ];
        if ($system) {
            $body['systemInstruction'] = ['parts' => [['text' => $system]]];
        }

        try {
            $headers = [
                'Content-Type' => 'application/json',
                'Authorization' => 'Bearer '.$accessToken
            ];
            $query = [];
            
            \Log::info('Vertex AI Request', [
                'endpoint' => $endpoint,
                'query' => $query,
                'body' => $body
            ]);
            
            $response = $this->http->post($endpoint, [
                'headers' => $headers,
                'query' => $query,
                'json' => $body,
            ]);
            $data = json_decode((string) $response->getBody(), true);
            if ($response->getStatusCode() < 200 || $response->getStatusCode() >= 300) {
                \Log::error('Vertex AI Non-200 response', [
                    'status' => $response->getStatusCode(),
                    'data' => $data
                ]);
                return ['text' => null, 'json' => null];
            }
            
            \Log::info('Vertex AI Response', [
                'status' => $response->getStatusCode(),
                'data' => $data
            ]);
            
            $text = $data['candidates'][0]['content']['parts'][0]['text'] ?? null;
            $json = null;
            if (is_string($text)) {
                $json = json_decode($text, true);
            }
            return ['text' => $text, 'json' => $json];
        } catch (GuzzleException $e) {
            \Log::error('Vertex AI Error', [
                'error' => $e->getMessage(),
                'endpoint' => $endpoint,
                'query' => $query ?? []
            ]);
            return ['text' => null, 'json' => null];
        }
    }
}


