<?php

namespace App\Services\AI;

use Google\Auth\ApplicationDefaultCredentials;
use Google\Auth\Credentials\CredentialsLoader;
use Google\Auth\Credentials\ServiceAccountCredentials;
use GuzzleHttp\Client as HttpClient;
use GuzzleHttp\Exception\GuzzleException;

class VertexClient
{
    private string $project;
    private string $location;
    private HttpClient $http;

    public function __construct()
    {
        $this->project = config('vertex.project');
        $this->location = config('vertex.location');
        $this->http = new HttpClient([
            'base_uri' => sprintf('https://%s-aiplatform.googleapis.com', $this->location),
            'timeout' => 15,
        ]);
    }

    private function getAccessToken(): ?string
    {
        $scopes = ['https://www.googleapis.com/auth/cloud-platform'];
        if ($json = env('GOOGLE_APPLICATION_CREDENTIALS_JSON')) {
            $creds = ServiceAccountCredentials::fromJson(json_decode($json, true));
            $creds->setScopes($scopes);
            $token = $creds->fetchAuthToken();
            return $token['access_token'] ?? null;
        }
        $creds = ApplicationDefaultCredentials::getCredentials($scopes);
        $token = $creds->fetchAuthToken();
        return $token['access_token'] ?? null;
    }

    public function generate(string $model, string $prompt, ?string $system = null, array $params = []): array
    {
        $accessToken = $this->getAccessToken();
        if (!$accessToken || !$this->project || !$this->location) {
            return ['text' => null, 'json' => null];
        }

        $endpoint = sprintf('/v1/projects/%s/locations/%s/publishers/google/models/%s:generateContent', $this->project, $this->location, $model);

        $body = [
            'contents' => [
                [
                    'role' => 'user',
                    'parts' => [['text' => $prompt]],
                ],
            ],
            'generationConfig' => array_merge([
                'temperature' => (float) config('vertex.temperature', 0.2),
                'topP' => (float) config('vertex.top_p', 0.95),
                'topK' => (int) config('vertex.top_k', 40),
            ], $params),
        ];
        if ($system) {
            $body['systemInstruction'] = ['parts' => [['text' => $system]]];
        }

        try {
            $response = $this->http->post($endpoint, [
                'headers' => [
                    'Authorization' => 'Bearer '.$accessToken,
                    'Content-Type' => 'application/json',
                ],
                'json' => $body,
            ]);
            $data = json_decode((string) $response->getBody(), true);
            $text = $data['candidates'][0]['content']['parts'][0]['text'] ?? null;
            $json = null;
            if (is_string($text)) {
                $json = json_decode($text, true);
            }
            return ['text' => $text, 'json' => $json];
        } catch (GuzzleException $e) {
            return ['text' => null, 'json' => null];
        }
    }
}


