<?php

namespace App\Services\AI;

use Google\Auth\ApplicationDefaultCredentials;
use Google\Auth\Credentials\CredentialsLoader;
use Google\Auth\Credentials\ServiceAccountCredentials;
use GuzzleHttp\Client as HttpClient;
use GuzzleHttp\Exception\GuzzleException;

class VertexClient
{
    private string $project;
    private string $location;
    private HttpClient $http;
    private ?string $apiKey = null;

    public function __construct()
    {
        $this->project = (string) (config('vertex.project') ?? '');
        $this->location = (string) (config('vertex.location') ?? 'us-central1');
        $this->apiKey = config('vertex.api_key');
        
        // Use location-specific endpoint for Vertex AI
        $this->http = new HttpClient([
            'base_uri' => sprintf('https://%s-aiplatform.googleapis.com', $this->location),
            'timeout' => config('vertex.timeout', 120), // Configurable timeout for OCR requests with images
        ]);
    }

    private function getAccessToken(): ?string
    {
        $scopes = ['https://www.googleapis.com/auth/cloud-platform'];
        try {
            if ($json = env('GOOGLE_APPLICATION_CREDENTIALS_JSON')) {
                $creds = ServiceAccountCredentials::fromJson(json_decode($json, true));
                $creds->setScopes($scopes);
                $token = $creds->fetchAuthToken();
                return $token['access_token'] ?? null;
            }
            $creds = ApplicationDefaultCredentials::getCredentials($scopes);
            $token = $creds->fetchAuthToken();
            return $token['access_token'] ?? null;
        } catch (\Throwable $e) {
            return null;
        }
    }

    public function generate(string $model, string $prompt, ?string $system = null, array $params = []): array
    {
        if (!$this->project || !$this->location) {
            return ['text' => null, 'json' => null];
        }

        // Get OAuth2 access token
        $accessToken = $this->getAccessToken();
        if (!$accessToken) {
            \Log::error('Vertex AI: No access token available');
            return ['text' => null, 'json' => null];
        }

        // Use Vertex AI endpoint format as per Google Cloud documentation
        $this->http = new HttpClient([
            'base_uri' => sprintf('https://%s-aiplatform.googleapis.com', $this->location),
            'timeout' => config('vertex.timeout', 120), // Configurable timeout for OCR requests with images
        ]);
        $endpoint = sprintf('/v1/projects/%s/locations/%s/publishers/google/models/%s:generateContent', $this->project, $this->location, $model);

        // Parse the prompt to check if it contains image data
        $promptData = json_decode($prompt, true);
        $parts = [['text' => $prompt]];
        
        if (is_array($promptData) && isset($promptData['image'])) {
            // Handle multimodal input with image
            $parts = [
                ['text' => $promptData['instructions'] ?? $prompt],
                ['inline_data' => [
                    'mime_type' => 'image/jpeg',
                    'data' => $promptData['image']
                ]]
            ];
        }

        $body = [
            'contents' => [
                [
                    'role' => 'user',
                    'parts' => $parts,
                ],
            ],
            'generationConfig' => array_merge([
                'temperature' => (float) config('vertex.temperature', 0.2),
                'topP' => (float) config('vertex.top_p', 0.95),
                'topK' => (int) config('vertex.top_k', 40),
            ], $params),
        ];
        if ($system) {
            $body['systemInstruction'] = ['parts' => [['text' => $system]]];
        }

        try {
            $headers = [
                'Content-Type' => 'application/json',
                'Authorization' => 'Bearer '.$accessToken
            ];
            $query = [];
            
            \Log::info('Vertex AI Request', [
                'endpoint' => $endpoint,
                'query' => $query,
                'body' => $body
            ]);
            
            $response = $this->http->post($endpoint, [
                'headers' => $headers,
                'query' => $query,
                'json' => $body,
            ]);
            $data = json_decode((string) $response->getBody(), true);
            
            \Log::info('Vertex AI Response', [
                'status' => $response->getStatusCode(),
                'data' => $data
            ]);
            
            $text = $data['candidates'][0]['content']['parts'][0]['text'] ?? null;
            $json = null;
            if (is_string($text)) {
                $json = json_decode($text, true);
            }
            return ['text' => $text, 'json' => $json];
        } catch (GuzzleException $e) {
            \Log::error('Vertex AI Error', [
                'error' => $e->getMessage(),
                'endpoint' => $endpoint,
                'query' => $query ?? []
            ]);
            return ['text' => null, 'json' => null];
        }
    }
}


