<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class ExftyHistoryController extends Controller
{
    public function getHistory($shipmentLineId)
    {
        try {
            // Validate shipment line ID
            if (!is_numeric($shipmentLineId) || $shipmentLineId <= 0) {
                return response()->json(['history' => []]);
            }

            $history = DB::table('audits')
                ->where('auditable_type', 'App\\Models\\ShipmentLine')
                ->where('auditable_id', $shipmentLineId)
                ->whereRaw("JSON_CONTAINS_PATH(old_values, 'one', '$.exfty')")
                ->select([
                    'created_at',
                    DB::raw("JSON_UNQUOTE(JSON_EXTRACT(old_values, '$.exfty')) as old_exfty"),
                    DB::raw("JSON_UNQUOTE(JSON_EXTRACT(new_values, '$.exfty')) as new_exfty")
                ])
                ->orderBy('created_at', 'asc')
                ->limit(20) // Limit to prevent memory issues
                ->get();

            // Add current exfty as the latest entry if it exists
            $currentExfty = DB::table('shipment_lines')
                ->where('id', $shipmentLineId)
                ->value('exfty');
                
            if ($currentExfty) {
                $history->push((object) [
                    'created_at' => now(),
                    'old_exfty' => $history->last()?->new_exfty ?? null,
                    'new_exfty' => $currentExfty
                ]);
            }

            // Format the response
            $formattedHistory = $history->map(function($change) use ($history) {
                return [
                    'date' => $change->new_exfty ? date_format(date_create($change->new_exfty), "d M") : null,
                    'is_current' => $change === $history->last()
                ];
            })->filter(function($item) {
                return $item['date'] !== null;
            })->values();

            return response()->json(['history' => $formattedHistory]);
        } catch (\Exception $e) {
            \Log::error('Error fetching exfty history: ' . $e->getMessage());
            return response()->json(['history' => []]);
        }
    }
}
