<?php

namespace App\Livewire\Admin;

use App\Models\Season;
use Livewire\Component;
use Livewire\WithPagination;

class SeasonDetail extends Component
{
    use WithPagination;
    
    public Season $season;
    public $editing = false;
    
    // Form fields for editing
    public $name;
    public $slug;
    public $description;
    public $startDate;
    public $endDate;
    public $image;
    public $isActive;
    public $sortOrder;
    
    protected $rules = [
        'name' => 'required|string|max:255',
        'slug' => 'required|string|max:255|unique:seasons,slug',
        'description' => 'nullable|string',
        'startDate' => 'nullable|date',
        'endDate' => 'nullable|date|after_or_equal:startDate',
        'image' => 'nullable|string|max:255',
        'isActive' => 'boolean',
        'sortOrder' => 'required|integer|min:0',
    ];
    
    public function mount($seasonId)
    {
        $this->season = Season::findOrFail($seasonId);
        $this->loadFormData();
    }
    
    public function loadFormData()
    {
        $this->name = $this->season->name;
        $this->slug = $this->season->slug;
        $this->description = $this->season->description;
        $this->startDate = $this->season->start_date?->format('Y-m-d');
        $this->endDate = $this->season->end_date?->format('Y-m-d');
        $this->image = $this->season->image;
        $this->isActive = $this->season->is_active;
        $this->sortOrder = $this->season->sort_order;
    }
    
    public function toggleEdit()
    {
        if ($this->editing) {
            $this->loadFormData(); // Reset form data
        }
        $this->editing = !$this->editing;
    }
    
    public function save()
    {
        // Update validation rules to exclude current season from unique check
        $this->rules['slug'] = 'required|string|max:255|unique:seasons,slug,' . $this->season->id;
        
        $this->validate();
        
        $this->season->update([
            'name' => $this->name,
            'slug' => $this->slug,
            'description' => $this->description,
            'start_date' => $this->startDate ?: null,
            'end_date' => $this->endDate ?: null,
            'image' => $this->image,
            'is_active' => $this->isActive,
            'sort_order' => $this->sortOrder,
        ]);
        
        $this->editing = false;
        session()->flash('message', 'Season updated successfully!');
    }
    
    public function cancel()
    {
        $this->loadFormData();
        $this->editing = false;
    }
    
    public function render()
    {
        // Get items for this season (when Item model is created)
        $items = collect(); // Empty collection for now
        
        return view('livewire.admin.season-detail', [
            'items' => $items
        ])
            ->layout('layouts.admin')
            ->title('Season Details');
    }
}
