<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    /**
     * Run the migrations.
     */
    public function up(): void
    {
        // Add composite indexes for common query patterns in shipment schedule
        
        // 1. Customer Order Line Quantities - for our detailed query joins
        Schema::table('customer_order_line_quantities', function (Blueprint $table) {
            // Composite index for customer_order_lines_id + sizes_id (very common in our query)
            $table->index(['customer_order_lines_id', 'sizes_id'], 'idx_colq_col_sizes');
        });
        
        // 2. Shipment Line Sizes - for our detailed query joins
        Schema::table('shipment_line_sizes', function (Blueprint $table) {
            // Composite index for shipment_line_id + sizes_id (very common in our query)
            $table->index(['shipment_line_id', 'sizes_id'], 'idx_sls_sl_sizes');
        });
        
        // 3. Samples - for our sample data queries
        Schema::table('samples', function (Blueprint $table) {
            // Composite index for colourways_id + sample_types_id + deleted_at (our sample query pattern)
            $table->index(['colourways_id', 'sample_types_id', 'deleted_at'], 'idx_samples_cw_type_deleted');
            // Index for created_at to optimize the MAX(created_at) subquery
            $table->index(['colourways_id', 'sample_types_id', 'created_at'], 'idx_samples_cw_type_created');
        });
        
        // 4. Customer Orders - for main query filtering
        Schema::table('customer_orders', function (Blueprint $table) {
            // Composite index for order_type + cancelled (very common filter)
            $table->index(['order_type', 'cancelled'], 'idx_co_type_cancelled');
            // Composite index for customers_id + seasons_id (common filter combination)
            $table->index(['customers_id', 'seasons_id'], 'idx_co_customer_season');
        });
        
        // 5. Shipment Lines - for main query filtering and ordering
        Schema::table('shipment_lines', function (Blueprint $table) {
            // Composite index for complete + exfty (common filter and order combination)
            $table->index(['complete', 'exfty'], 'idx_sl_complete_exfty');
            // Composite index for shipment_id + complete (for shipment grouping)
            $table->index(['shipment_id', 'complete'], 'idx_sl_shipment_complete');
        });
        
        // 6. Price Resolutions - add fresh_at to existing unique constraint for better performance
        Schema::table('price_resolutions', function (Blueprint $table) {
            // Index for fresh_at to optimize the NOT NULL condition
            $table->index(['fresh_at'], 'idx_pr_fresh_at');
            // Composite index for our most common query pattern
            $table->index(['colourways_id', 'sizes_id', 'phase_id', 'season_id', 'fresh_at'], 'idx_pr_cw_size_phase_season_fresh');
        });
    }

    /**
     * Reverse the migrations.
     */
    public function down(): void
    {
        // Drop the indexes in reverse order
        Schema::table('price_resolutions', function (Blueprint $table) {
            $table->dropIndex('idx_pr_fresh_at');
            $table->dropIndex('idx_pr_cw_size_phase_season_fresh');
        });
        
        Schema::table('shipment_lines', function (Blueprint $table) {
            $table->dropIndex('idx_sl_complete_exfty');
            $table->dropIndex('idx_sl_shipment_complete');
        });
        
        Schema::table('customer_orders', function (Blueprint $table) {
            $table->dropIndex('idx_co_type_cancelled');
            $table->dropIndex('idx_co_customer_season');
        });
        
        Schema::table('samples', function (Blueprint $table) {
            $table->dropIndex('idx_samples_cw_type_deleted');
            $table->dropIndex('idx_samples_cw_type_created');
        });
        
        Schema::table('shipment_line_sizes', function (Blueprint $table) {
            $table->dropIndex('idx_sls_sl_sizes');
        });
        
        Schema::table('customer_order_line_quantities', function (Blueprint $table) {
            $table->dropIndex('idx_colq_col_sizes');
        });
    }
};