<?php

namespace App\Http\Livewire\Dashboard;

use App\Models\Samples;
use App\Models\ShipmentLine;
use App\Models\ShipmentLineSizes;
use App\Models\CustomerOrderLineQuantities;
use Livewire\Component;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Cache;

class DashboardMetrics extends Component
{
    public $totalQmtsShipped = 0;
    public $totalGmtsLeftToShip = 0;
    public $totalValueShipped = 0;
    public $totalSamplesSent = 0;
    public $totalInvoicesSent = 0;
    public $isLoading = true;

    private const CACHE_DURATION = 24 * 60; // 24 hours in minutes

    public function mount()
    {
        $this->loadMetrics();
    }

    public function refreshData()
    {
        $this->isLoading = true;
        $this->loadMetrics();
        $this->isLoading = false;
    }

    public function loadMetrics()
    {
        // Load all metrics with caching
        $this->totalQmtsShipped = $this->getTotalQmtsShipped();
        $this->totalGmtsLeftToShip = $this->getTotalGmtsLeftToShip();
        $this->totalValueShipped = $this->getTotalValueShipped();
        $this->totalSamplesSent = $this->getTotalSamplesSent();
        $this->totalInvoicesSent = $this->getTotalInvoicesSent();
        
        $this->isLoading = false;
    }

    private function getTotalQmtsShipped()
    {
        return Cache::remember('dashboard_total_qmts_shipped', self::CACHE_DURATION, function () {
            return ShipmentLineSizes::whereNotNull('shipped_qty')
                ->where('shipped_qty', '>', 0)
                ->whereNull('deleted_at')
                ->sum('shipped_qty');
        });
    }

    private function getTotalGmtsLeftToShip()
    {
        return Cache::remember('dashboard_total_gmts_left_to_ship', self::CACHE_DURATION, function () {
            // Get total ordered quantities
            $totalOrdered = CustomerOrderLineQuantities::whereNull('deleted_at')
                ->sum('qty');

            // Get total shipped quantities
            $totalShipped = ShipmentLineSizes::whereNotNull('shipped_qty')
                ->where('shipped_qty', '>', 0)
                ->whereNull('deleted_at')
                ->sum('shipped_qty');

            return max(0, $totalOrdered - $totalShipped);
        });
    }

    private function getTotalValueShipped()
    {
        return Cache::remember('dashboard_total_value_shipped', self::CACHE_DURATION, function () {
            $totalValue = 0;

            // Get all shipped quantities with their order information
            $shippedQuantities = ShipmentLineSizes::with([
                'shipment_lines.customer_order_lines.customer_orders',
                'shipment_lines.customer_order_lines.customer_order_line_quantities'
            ])
            ->whereNotNull('shipped_qty')
            ->where('shipped_qty', '>', 0)
            ->whereNull('deleted_at')
            ->whereHas('shipment_lines.customer_order_lines.customer_orders')
            ->get();

            foreach ($shippedQuantities as $shippedQty) {
                $shipmentLine = $shippedQty->shipment_lines;
                
                // Check if shipment line and customer order lines exist
                if (!$shipmentLine || !$shipmentLine->customer_order_lines) {
                    continue;
                }
                
                $order = $shipmentLine->customer_order_lines->customer_orders;
                
                // Check if customer orders exist
                if (!$order) {
                    continue;
                }
                
                $isWholesale = $order->order_type === 'wholesale';

                // Find the corresponding order line quantity for pricing
                $orderLineQty = $shipmentLine->customer_order_lines->customer_order_line_quantities
                    ->where('sizes_id', $shippedQty->sizes_id)
                    ->first();

                if ($orderLineQty) {
                    if ($isWholesale) {
                        // Use price cache for wholesale
                        $priceModel = $orderLineQty->price_model;
                        if ($priceModel && isset($priceModel['quote_base'])) {
                            $totalValue += $priceModel['quote_base'] * $shippedQty->shipped_qty;
                        }
                    } else {
                        // Use price column for commission
                        $totalValue += $orderLineQty->price * $shippedQty->shipped_qty;
                    }
                }
            }

            return $totalValue;
        });
    }

    private function getTotalSamplesSent()
    {
        return Cache::remember('dashboard_total_samples_sent', self::CACHE_DURATION, function () {
            return Samples::whereNotNull('date_sent')
                ->whereNull('deleted_at')
                ->count();
        });
    }

    private function getTotalInvoicesSent()
    {
        return Cache::remember('dashboard_total_invoices_sent', self::CACHE_DURATION, function () {
            // Get distinct rt_invoice from samples
            $samplesInvoices = Samples::whereNotNull('rt_invoice')
                ->where('rt_invoice', '!=', '')
                ->whereNull('deleted_at')
                ->distinct()
                ->pluck('rt_invoice')
                ->count();

            // Get distinct rt_invoice from shipment_lines
            $shipmentInvoices = ShipmentLine::whereNotNull('rt_invoice')
                ->where('rt_invoice', '!=', '')
                ->whereNull('deleted_at')
                ->distinct()
                ->pluck('rt_invoice')
                ->count();

            return $samplesInvoices + $shipmentInvoices;
        });
    }

    public function render()
    {
        return view('livewire.dashboard.dashboard-metrics');
    }
}
