<?php

namespace App\Http\Livewire\Dashboard;

use App\Models\Samples;
use App\Models\ShipmentLine;
use App\Models\ShipmentLineSizes;
use App\Models\CustomerOrderLineQuantities;
use Livewire\Component;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Cache;

class DashboardMetrics extends Component
{
    public $totalStylesInSeason = 0;
    public $totalQmtsShipped = 0;
    public $totalGmtsLeftToShip = 0;
    public $totalValueShipped = 0;
    public $totalValueOrdered = 0;
    public $valueShippedPercentage = 0;
    public $totalSamplesSent = 0;
    public $totalInvoicesSent = 0;
    public $isLoading = true;
    public $selectedSeason = null;
    public $selectedOrderType = 'all';

    private const CACHE_DURATION = 24 * 60; // 24 hours in minutes

    protected $listeners = [
        'dashboardFiltersUpdated' => 'updateFilters',
        'refreshDashboard' => 'refreshData'
    ];

    public function mount()
    {
        // Get global dashboard filter values
        $this->selectedSeason = auth()->user()->setting('dashboard_season_filter');
        $this->selectedOrderType = auth()->user()->setting('dashboard_order_type_filter', 'all');
        
        $this->loadMetrics();
    }

    public function updateFilters($filters)
    {
        $this->selectedSeason = $filters['season'];
        $this->selectedOrderType = $filters['orderType'];
        $this->isLoading = true;
        $this->loadMetrics();
        $this->isLoading = false;
    }

    public function refreshData()
    {
        $this->isLoading = true;
        $this->loadMetrics();
        $this->isLoading = false;
    }

    public function loadMetrics()
    {
        \Log::info("DashboardMetrics: Starting to load metrics");
        
        // Load all metrics with caching
        $this->totalStylesInSeason = $this->getTotalStylesInSeason();
        \Log::info("DashboardMetrics: Total Styles in Season = " . $this->totalStylesInSeason);
        
        $this->totalQmtsShipped = $this->getTotalQmtsShipped();
        \Log::info("DashboardMetrics: Total GMTs Shipped = " . $this->totalQmtsShipped);
        
        $this->totalGmtsLeftToShip = $this->getTotalGmtsLeftToShip();
        \Log::info("DashboardMetrics: Total GMTs Left to Ship = " . $this->totalGmtsLeftToShip);
        
        $this->totalValueShipped = $this->getTotalValueShipped();
        \Log::info("DashboardMetrics: Total Value Shipped = " . $this->totalValueShipped);
        
        $this->totalValueOrdered = $this->getTotalValueOrdered();
        \Log::info("DashboardMetrics: Total Value Ordered = " . $this->totalValueOrdered);
        
        // Calculate percentage
        if ($this->totalValueOrdered > 0) {
            $this->valueShippedPercentage = round(($this->totalValueShipped / $this->totalValueOrdered) * 100, 1);
        } else {
            $this->valueShippedPercentage = 0;
        }
        \Log::info("DashboardMetrics: Value Shipped Percentage = " . $this->valueShippedPercentage . "%");
        
        $this->totalSamplesSent = $this->getTotalSamplesSent();
        \Log::info("DashboardMetrics: Total Samples Sent = " . $this->totalSamplesSent);
        
        $this->totalInvoicesSent = $this->getTotalInvoicesSent();
        \Log::info("DashboardMetrics: Total Invoices Sent = " . $this->totalInvoicesSent);
        
        $this->isLoading = false;
        \Log::info("DashboardMetrics: Finished loading metrics");
    }

    private function getTotalStylesInSeason()
    {
        $cacheKey = "dashboard_total_styles_in_orders_season_{$this->selectedSeason}_order_type_{$this->selectedOrderType}";
        return Cache::remember($cacheKey, self::CACHE_DURATION, function () {
            if (!$this->selectedSeason) {
                return 0;
            }

            $query = DB::table('styles')
                ->join('style_versions', 'styles.id', '=', 'style_versions.styles_id')
                ->join('colourways', 'style_versions.id', '=', 'colourways.style_versions_id')
                ->join('customer_order_lines', 'colourways.id', '=', 'customer_order_lines.colourways_id')
                ->join('customer_orders', 'customer_order_lines.customer_orders_id', '=', 'customer_orders.id')
                ->where('styles.seasons_id', $this->selectedSeason)
                ->whereNull('styles.deleted_at')
                ->where('styles.cancelled', false)
                ->whereNull('style_versions.deleted_at')
                ->whereNull('colourways.deleted_at')
                ->where('colourways.cancelled', false)
                ->whereNull('customer_order_lines.deleted_at')
                ->where('customer_order_lines.cancelled', false)
                ->whereNull('customer_orders.deleted_at')
                ->where('customer_orders.cancelled', false);

            // Apply order type filter
            if ($this->selectedOrderType !== 'all') {
                $query->where('customer_orders.order_type', $this->selectedOrderType);
            }

            return $query->distinct('styles.id')->count('styles.id');
        });
    }

    private function getTotalQmtsShipped()
    {
        $cacheKey = "dashboard_total_qmts_shipped_season_{$this->selectedSeason}_order_type_{$this->selectedOrderType}";
        return Cache::remember($cacheKey, self::CACHE_DURATION, function () {
            $query = DB::table('shipment_line_sizes')
                ->join('shipment_lines', 'shipment_line_sizes.shipment_line_id', '=', 'shipment_lines.id')
                ->join('customer_order_lines', 'shipment_lines.customer_order_lines_id', '=', 'customer_order_lines.id')
                ->join('customer_orders', 'customer_order_lines.customer_orders_id', '=', 'customer_orders.id')
                ->whereNotNull('shipment_line_sizes.shipped_qty')
                ->where('shipment_line_sizes.shipped_qty', '>', 0)
                ->whereNull('shipment_line_sizes.deleted_at')
                ->whereNull('shipment_lines.deleted_at')
                ->whereNull('customer_order_lines.deleted_at')
                ->whereNull('customer_orders.deleted_at')
                ->where('customer_orders.cancelled', false)
                ->where('customer_order_lines.cancelled', false);

            // Apply season filter
            if ($this->selectedSeason) {
                $query->where('customer_orders.seasons_id', $this->selectedSeason);
            }

            // Apply order type filter
            if ($this->selectedOrderType !== 'all') {
                $query->where('customer_orders.order_type', $this->selectedOrderType);
            }

            return $query->sum('shipment_line_sizes.shipped_qty');
        });
    }

    private function getTotalGmtsLeftToShip()
    {
        $cacheKey = "dashboard_total_gmts_left_to_ship_season_{$this->selectedSeason}_order_type_{$this->selectedOrderType}";
        return Cache::remember($cacheKey, self::CACHE_DURATION, function () {
            // Get total ordered quantities (only from non-cancelled, non-deleted orders)
            $totalOrderedQuery = DB::table('customer_order_line_quantities')
                ->join('customer_order_lines', 'customer_order_line_quantities.customer_order_lines_id', '=', 'customer_order_lines.id')
                ->join('customer_orders', 'customer_order_lines.customer_orders_id', '=', 'customer_orders.id')
                ->whereNull('customer_order_line_quantities.deleted_at')
                ->whereNull('customer_order_lines.deleted_at')
                ->whereNull('customer_orders.deleted_at')
                ->where('customer_orders.cancelled', false)
                ->where('customer_order_lines.cancelled', false);

            // Apply season filter
            if ($this->selectedSeason) {
                $totalOrderedQuery->where('customer_orders.seasons_id', $this->selectedSeason);
            }

            // Apply order type filter
            if ($this->selectedOrderType !== 'all') {
                $totalOrderedQuery->where('customer_orders.order_type', $this->selectedOrderType);
            }

            $totalOrdered = $totalOrderedQuery->sum('customer_order_line_quantities.qty');

            // Get total shipped quantities (only from non-cancelled, non-deleted orders)
            $totalShippedQuery = DB::table('shipment_line_sizes')
                ->join('shipment_lines', 'shipment_line_sizes.shipment_line_id', '=', 'shipment_lines.id')
                ->join('customer_order_lines', 'shipment_lines.customer_order_lines_id', '=', 'customer_order_lines.id')
                ->join('customer_orders', 'customer_order_lines.customer_orders_id', '=', 'customer_orders.id')
                ->whereNotNull('shipment_line_sizes.shipped_qty')
                ->where('shipment_line_sizes.shipped_qty', '>', 0)
                ->whereNull('shipment_line_sizes.deleted_at')
                ->whereNull('shipment_lines.deleted_at')
                ->whereNull('customer_order_lines.deleted_at')
                ->whereNull('customer_orders.deleted_at')
                ->where('customer_orders.cancelled', false)
                ->where('customer_order_lines.cancelled', false);

            // Apply season filter
            if ($this->selectedSeason) {
                $totalShippedQuery->where('customer_orders.seasons_id', $this->selectedSeason);
            }

            // Apply order type filter
            if ($this->selectedOrderType !== 'all') {
                $totalShippedQuery->where('customer_orders.order_type', $this->selectedOrderType);
            }

            $totalShipped = $totalShippedQuery->sum('shipment_line_sizes.shipped_qty');

            return max(0, $totalOrdered - $totalShipped);
        });
    }

    private function getTotalValueShipped()
    {
        $cacheKey = "dashboard_total_value_shipped_season_{$this->selectedSeason}_order_type_{$this->selectedOrderType}";
        return Cache::remember($cacheKey, self::CACHE_DURATION, function () {
            // Calculate wholesale value using SQL
            $wholesaleQuery = DB::table('shipment_line_sizes')
                ->join('shipment_lines', 'shipment_line_sizes.shipment_line_id', '=', 'shipment_lines.id')
                ->join('customer_order_lines', 'shipment_lines.customer_order_lines_id', '=', 'customer_order_lines.id')
                ->join('customer_orders', 'customer_order_lines.customer_orders_id', '=', 'customer_orders.id')
                ->join('customer_order_line_quantities', function($join) {
                    $join->on('customer_order_line_quantities.customer_order_lines_id', '=', 'customer_order_lines.id')
                         ->on('customer_order_line_quantities.sizes_id', '=', 'shipment_line_sizes.sizes_id');
                })
                ->join('colourways', 'customer_order_lines.colourways_id', '=', 'colourways.id')
                ->join('style_versions', 'colourways.style_versions_id', '=', 'style_versions.id')
                ->join('price_resolutions', function($join) {
                    $join->on('price_resolutions.style_versions_id', '=', 'style_versions.id')
                         ->on('price_resolutions.colourways_id', '=', 'colourways.id')
                         ->on('price_resolutions.sizes_id', '=', 'shipment_line_sizes.sizes_id');
                })
                ->whereNotNull('shipment_line_sizes.shipped_qty')
                ->where('shipment_line_sizes.shipped_qty', '>', 0)
                ->whereNull('shipment_line_sizes.deleted_at')
                ->whereNull('shipment_lines.deleted_at')
                ->whereNull('customer_order_lines.deleted_at')
                ->whereNull('customer_orders.deleted_at')
                ->where('customer_orders.cancelled', false)
                ->where('customer_order_lines.cancelled', false)
                ->where('customer_orders.order_type', 'wholesale');

            // Apply season filter
            if ($this->selectedSeason) {
                $wholesaleQuery->where('customer_orders.seasons_id', $this->selectedSeason);
            }

            // Apply order type filter - only include wholesale if order type is 'all' or 'wholesale'
            if ($this->selectedOrderType === 'commission') {
                $wholesaleValue = 0;
            } else {
                $wholesaleValue = $wholesaleQuery->selectRaw('SUM(price_resolutions.quote_base * shipment_line_sizes.shipped_qty) as total')
                    ->value('total') ?? 0;
            }

            // Calculate commission value using SQL
            $commissionQuery = DB::table('shipment_line_sizes')
                ->join('shipment_lines', 'shipment_line_sizes.shipment_line_id', '=', 'shipment_lines.id')
                ->join('customer_order_lines', 'shipment_lines.customer_order_lines_id', '=', 'customer_order_lines.id')
                ->join('customer_orders', 'customer_order_lines.customer_orders_id', '=', 'customer_orders.id')
                ->join('customer_order_line_quantities', function($join) {
                    $join->on('customer_order_line_quantities.customer_order_lines_id', '=', 'customer_order_lines.id')
                         ->on('customer_order_line_quantities.sizes_id', '=', 'shipment_line_sizes.sizes_id');
                })
                ->whereNotNull('shipment_line_sizes.shipped_qty')
                ->where('shipment_line_sizes.shipped_qty', '>', 0)
                ->whereNull('shipment_line_sizes.deleted_at')
                ->whereNull('shipment_lines.deleted_at')
                ->whereNull('customer_order_lines.deleted_at')
                ->whereNull('customer_orders.deleted_at')
                ->where('customer_orders.cancelled', false)
                ->where('customer_order_lines.cancelled', false)
                ->where('customer_orders.order_type', 'commission');

            // Apply season filter
            if ($this->selectedSeason) {
                $commissionQuery->where('customer_orders.seasons_id', $this->selectedSeason);
            }

            // Apply order type filter - only include commission if order type is 'all' or 'commission'
            if ($this->selectedOrderType === 'wholesale') {
                $commissionValue = 0;
            } else {
                $commissionValue = $commissionQuery->selectRaw('SUM(customer_order_line_quantities.price * shipment_line_sizes.shipped_qty) as total')
                    ->value('total') ?? 0;
            }

            return $wholesaleValue + $commissionValue;
        });
    }

    private function getTotalValueOrdered()
    {
        $cacheKey = "dashboard_total_value_ordered_season_{$this->selectedSeason}_order_type_{$this->selectedOrderType}";
        return Cache::remember($cacheKey, self::CACHE_DURATION, function () {
            // Calculate wholesale value ordered using SQL
            $wholesaleQuery = DB::table('customer_order_line_quantities')
                ->join('customer_order_lines', 'customer_order_line_quantities.customer_order_lines_id', '=', 'customer_order_lines.id')
                ->join('customer_orders', 'customer_order_lines.customer_orders_id', '=', 'customer_orders.id')
                ->join('colourways', 'customer_order_lines.colourways_id', '=', 'colourways.id')
                ->join('style_versions', 'colourways.style_versions_id', '=', 'style_versions.id')
                ->join('price_resolutions', function($join) {
                    $join->on('price_resolutions.style_versions_id', '=', 'style_versions.id')
                         ->on('price_resolutions.colourways_id', '=', 'colourways.id')
                         ->on('price_resolutions.sizes_id', '=', 'customer_order_line_quantities.sizes_id');
                })
                ->whereNull('customer_order_line_quantities.deleted_at')
                ->whereNull('customer_order_lines.deleted_at')
                ->whereNull('customer_orders.deleted_at')
                ->where('customer_orders.cancelled', false)
                ->where('customer_order_lines.cancelled', false)
                ->where('customer_orders.order_type', 'wholesale');

            // Apply season filter
            if ($this->selectedSeason) {
                $wholesaleQuery->where('customer_orders.seasons_id', $this->selectedSeason);
            }

            // Apply order type filter - only include wholesale if order type is 'all' or 'wholesale'
            if ($this->selectedOrderType === 'commission') {
                $wholesaleValue = 0;
            } else {
                $wholesaleValue = $wholesaleQuery->selectRaw('SUM(price_resolutions.quote_base * customer_order_line_quantities.qty) as total')
                    ->value('total') ?? 0;
            }

            // Calculate commission value ordered using SQL
            $commissionQuery = DB::table('customer_order_line_quantities')
                ->join('customer_order_lines', 'customer_order_line_quantities.customer_order_lines_id', '=', 'customer_order_lines.id')
                ->join('customer_orders', 'customer_order_lines.customer_orders_id', '=', 'customer_orders.id')
                ->whereNull('customer_order_line_quantities.deleted_at')
                ->whereNull('customer_order_lines.deleted_at')
                ->whereNull('customer_orders.deleted_at')
                ->where('customer_orders.cancelled', false)
                ->where('customer_order_lines.cancelled', false)
                ->where('customer_orders.order_type', 'commission');

            // Apply season filter
            if ($this->selectedSeason) {
                $commissionQuery->where('customer_orders.seasons_id', $this->selectedSeason);
            }

            // Apply order type filter - only include commission if order type is 'all' or 'commission'
            if ($this->selectedOrderType === 'wholesale') {
                $commissionValue = 0;
            } else {
                $commissionValue = $commissionQuery->selectRaw('SUM(customer_order_line_quantities.price * customer_order_line_quantities.qty) as total')
                    ->value('total') ?? 0;
            }

            return $wholesaleValue + $commissionValue;
        });
    }

    private function getTotalSamplesSent()
    {
        $cacheKey = "dashboard_total_samples_sent_season_{$this->selectedSeason}_order_type_{$this->selectedOrderType}";
        return Cache::remember($cacheKey, self::CACHE_DURATION, function () {
            $query = DB::table('samples')
                ->join('colourways', 'samples.colourways_id', '=', 'colourways.id')
                ->join('style_versions', 'colourways.style_versions_id', '=', 'style_versions.id')
                ->join('styles', 'style_versions.styles_id', '=', 'styles.id')
                ->whereNotNull('samples.date_sent')
                ->whereNull('samples.deleted_at')
                ->whereNull('colourways.deleted_at')
                ->where('colourways.cancelled', false);

            // Apply season filter
            if ($this->selectedSeason) {
                $query->where('styles.seasons_id', $this->selectedSeason);
            }

            // Apply order type filter - samples don't have direct order type, so we'll skip this filter for samples
            // Samples are associated with styles, not customer orders directly

            return $query->count();
        });
    }

    private function getTotalInvoicesSent()
    {
        $cacheKey = "dashboard_total_invoices_sent_season_{$this->selectedSeason}_order_type_{$this->selectedOrderType}";
        return Cache::remember($cacheKey, self::CACHE_DURATION, function () {
            // Get distinct rt_invoice from samples (only non-cancelled, non-deleted)
            $samplesQuery = DB::table('samples')
                ->join('colourways', 'samples.colourways_id', '=', 'colourways.id')
                ->join('style_versions', 'colourways.style_versions_id', '=', 'style_versions.id')
                ->join('styles', 'style_versions.styles_id', '=', 'styles.id')
                ->whereNotNull('samples.rt_invoice')
                ->where('samples.rt_invoice', '!=', '')
                ->whereNull('samples.deleted_at')
                ->whereNull('colourways.deleted_at')
                ->where('colourways.cancelled', false);

            // Apply season filter
            if ($this->selectedSeason) {
                $samplesQuery->where('styles.seasons_id', $this->selectedSeason);
            }

            // Apply order type filter - samples don't have direct order type, so we'll skip this filter for samples
            // Samples are associated with styles, not customer orders directly

            $samplesInvoices = $samplesQuery->distinct()->count('samples.rt_invoice');

            // Get distinct rt_invoice from shipment_lines (only non-cancelled, non-deleted)
            $shipmentQuery = DB::table('shipment_lines')
                ->join('customer_order_lines', 'shipment_lines.customer_order_lines_id', '=', 'customer_order_lines.id')
                ->join('customer_orders', 'customer_order_lines.customer_orders_id', '=', 'customer_orders.id')
                ->whereNotNull('shipment_lines.rt_invoice')
                ->where('shipment_lines.rt_invoice', '!=', '')
                ->whereNull('shipment_lines.deleted_at')
                ->whereNull('customer_order_lines.deleted_at')
                ->whereNull('customer_orders.deleted_at')
                ->where('customer_orders.cancelled', false)
                ->where('customer_order_lines.cancelled', false);

            // Apply season filter
            if ($this->selectedSeason) {
                $shipmentQuery->where('customer_orders.seasons_id', $this->selectedSeason);
            }

            // Apply order type filter
            if ($this->selectedOrderType !== 'all') {
                $shipmentQuery->where('customer_orders.order_type', $this->selectedOrderType);
            }

            $shipmentInvoices = $shipmentQuery->distinct()->count('shipment_lines.rt_invoice');

            return $samplesInvoices + $shipmentInvoices;
        });
    }

    public function render()
    {
        return view('livewire.dashboard.dashboard-metrics');
    }
}
