<?php

namespace App\Http\Livewire\Dashboard;

use App\Models\Samples;
use App\Models\ShipmentLine;
use App\Models\ShipmentLineSizes;
use App\Models\CustomerOrderLineQuantities;
use Livewire\Component;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Cache;

class DashboardMetrics extends Component
{
    public $totalQmtsShipped = 0;
    public $totalGmtsLeftToShip = 0;
    public $totalValueShipped = 0;
    public $totalSamplesSent = 0;
    public $totalInvoicesSent = 0;
    public $isLoading = true;

    private const CACHE_DURATION = 24 * 60; // 24 hours in minutes

    public function mount()
    {
        $this->loadMetrics();
    }

    public function refreshData()
    {
        $this->isLoading = true;
        $this->loadMetrics();
        $this->isLoading = false;
    }

    public function loadMetrics()
    {
        \Log::info("DashboardMetrics: Starting to load metrics");
        
        // Temporarily use hardcoded values for debugging
        $this->totalQmtsShipped = 1234567;
        \Log::info("DashboardMetrics: Total GMTs Shipped = " . $this->totalQmtsShipped);
        
        $this->totalGmtsLeftToShip = 987654;
        \Log::info("DashboardMetrics: Total GMTs Left to Ship = " . $this->totalGmtsLeftToShip);
        
        $this->totalValueShipped = 5000000;
        \Log::info("DashboardMetrics: Total Value Shipped = " . $this->totalValueShipped);
        
        $this->totalSamplesSent = 1234;
        \Log::info("DashboardMetrics: Total Samples Sent = " . $this->totalSamplesSent);
        
        $this->totalInvoicesSent = 567;
        \Log::info("DashboardMetrics: Total Invoices Sent = " . $this->totalInvoicesSent);
        
        $this->isLoading = false;
        \Log::info("DashboardMetrics: Finished loading metrics");
    }

    private function getTotalQmtsShipped()
    {
        return Cache::remember('dashboard_total_qmts_shipped', self::CACHE_DURATION, function () {
            return DB::table('shipment_line_sizes')
                ->join('shipment_lines', 'shipment_line_sizes.shipment_line_id', '=', 'shipment_lines.id')
                ->join('customer_order_lines', 'shipment_lines.customer_order_lines_id', '=', 'customer_order_lines.id')
                ->join('customer_orders', 'customer_order_lines.customer_orders_id', '=', 'customer_orders.id')
                ->whereNotNull('shipment_line_sizes.shipped_qty')
                ->where('shipment_line_sizes.shipped_qty', '>', 0)
                ->whereNull('shipment_line_sizes.deleted_at')
                ->whereNull('shipment_lines.deleted_at')
                ->whereNull('customer_order_lines.deleted_at')
                ->whereNull('customer_orders.deleted_at')
                ->where('customer_orders.cancelled', false)
                ->where('customer_order_lines.cancelled', false)
                ->sum('shipment_line_sizes.shipped_qty');
        });
    }

    private function getTotalGmtsLeftToShip()
    {
        return Cache::remember('dashboard_total_gmts_left_to_ship', self::CACHE_DURATION, function () {
            // Get total ordered quantities (only from non-cancelled, non-deleted orders)
            $totalOrdered = DB::table('customer_order_line_quantities')
                ->join('customer_order_lines', 'customer_order_line_quantities.customer_order_lines_id', '=', 'customer_order_lines.id')
                ->join('customer_orders', 'customer_order_lines.customer_orders_id', '=', 'customer_orders.id')
                ->whereNull('customer_order_line_quantities.deleted_at')
                ->whereNull('customer_order_lines.deleted_at')
                ->whereNull('customer_orders.deleted_at')
                ->where('customer_orders.cancelled', false)
                ->where('customer_order_lines.cancelled', false)
                ->sum('customer_order_line_quantities.qty');

            // Get total shipped quantities (only from non-cancelled, non-deleted orders)
            $totalShipped = DB::table('shipment_line_sizes')
                ->join('shipment_lines', 'shipment_line_sizes.shipment_line_id', '=', 'shipment_lines.id')
                ->join('customer_order_lines', 'shipment_lines.customer_order_lines_id', '=', 'customer_order_lines.id')
                ->join('customer_orders', 'customer_order_lines.customer_orders_id', '=', 'customer_orders.id')
                ->whereNotNull('shipment_line_sizes.shipped_qty')
                ->where('shipment_line_sizes.shipped_qty', '>', 0)
                ->whereNull('shipment_line_sizes.deleted_at')
                ->whereNull('shipment_lines.deleted_at')
                ->whereNull('customer_order_lines.deleted_at')
                ->whereNull('customer_orders.deleted_at')
                ->where('customer_orders.cancelled', false)
                ->where('customer_order_lines.cancelled', false)
                ->sum('shipment_line_sizes.shipped_qty');

            return max(0, $totalOrdered - $totalShipped);
        });
    }

    private function getTotalValueShipped()
    {
        return Cache::remember('dashboard_total_value_shipped', self::CACHE_DURATION, function () {
            // Calculate wholesale value using SQL
            $wholesaleValue = DB::table('shipment_line_sizes')
                ->join('shipment_lines', 'shipment_line_sizes.shipment_line_id', '=', 'shipment_lines.id')
                ->join('customer_order_lines', 'shipment_lines.customer_order_lines_id', '=', 'customer_order_lines.id')
                ->join('customer_orders', 'customer_order_lines.customer_orders_id', '=', 'customer_orders.id')
                ->join('customer_order_line_quantities', function($join) {
                    $join->on('customer_order_line_quantities.customer_order_lines_id', '=', 'customer_order_lines.id')
                         ->on('customer_order_line_quantities.sizes_id', '=', 'shipment_line_sizes.sizes_id');
                })
                ->join('colourways', 'customer_order_lines.colourways_id', '=', 'colourways.id')
                ->join('style_versions', 'colourways.style_versions_id', '=', 'style_versions.id')
                ->join('price_resolutions', function($join) {
                    $join->on('price_resolutions.style_versions_id', '=', 'style_versions.id')
                         ->on('price_resolutions.colourways_id', '=', 'colourways.id')
                         ->on('price_resolutions.sizes_id', '=', 'shipment_line_sizes.sizes_id');
                })
                ->whereNotNull('shipment_line_sizes.shipped_qty')
                ->where('shipment_line_sizes.shipped_qty', '>', 0)
                ->whereNull('shipment_line_sizes.deleted_at')
                ->whereNull('shipment_lines.deleted_at')
                ->whereNull('customer_order_lines.deleted_at')
                ->whereNull('customer_orders.deleted_at')
                ->where('customer_orders.cancelled', false)
                ->where('customer_order_lines.cancelled', false)
                ->where('customer_orders.order_type', 'wholesale')
                ->selectRaw('SUM(price_resolutions.quote_base * shipment_line_sizes.shipped_qty) as total')
                ->value('total') ?? 0;

            // Calculate commission value using SQL
            $commissionValue = DB::table('shipment_line_sizes')
                ->join('shipment_lines', 'shipment_line_sizes.shipment_line_id', '=', 'shipment_lines.id')
                ->join('customer_order_lines', 'shipment_lines.customer_order_lines_id', '=', 'customer_order_lines.id')
                ->join('customer_orders', 'customer_order_lines.customer_orders_id', '=', 'customer_orders.id')
                ->join('customer_order_line_quantities', function($join) {
                    $join->on('customer_order_line_quantities.customer_order_lines_id', '=', 'customer_order_lines.id')
                         ->on('customer_order_line_quantities.sizes_id', '=', 'shipment_line_sizes.sizes_id');
                })
                ->whereNotNull('shipment_line_sizes.shipped_qty')
                ->where('shipment_line_sizes.shipped_qty', '>', 0)
                ->whereNull('shipment_line_sizes.deleted_at')
                ->whereNull('shipment_lines.deleted_at')
                ->whereNull('customer_order_lines.deleted_at')
                ->whereNull('customer_orders.deleted_at')
                ->where('customer_orders.cancelled', false)
                ->where('customer_order_lines.cancelled', false)
                ->where('customer_orders.order_type', 'commission')
                ->selectRaw('SUM(customer_order_line_quantities.price * shipment_line_sizes.shipped_qty) as total')
                ->value('total') ?? 0;

            return $wholesaleValue + $commissionValue;
        });
    }

    private function getTotalSamplesSent()
    {
        return Cache::remember('dashboard_total_samples_sent', self::CACHE_DURATION, function () {
            return DB::table('samples')
                ->join('colourways', 'samples.colourways_id', '=', 'colourways.id')
                ->whereNotNull('samples.date_sent')
                ->whereNull('samples.deleted_at')
                ->whereNull('colourways.deleted_at')
                ->where('colourways.cancelled', false)
                ->count();
        });
    }

    private function getTotalInvoicesSent()
    {
        return Cache::remember('dashboard_total_invoices_sent', self::CACHE_DURATION, function () {
            // Get distinct rt_invoice from samples (only non-cancelled, non-deleted)
            $samplesInvoices = DB::table('samples')
                ->join('colourways', 'samples.colourways_id', '=', 'colourways.id')
                ->whereNotNull('samples.rt_invoice')
                ->where('samples.rt_invoice', '!=', '')
                ->whereNull('samples.deleted_at')
                ->whereNull('colourways.deleted_at')
                ->where('colourways.cancelled', false)
                ->distinct()
                ->count('samples.rt_invoice');

            // Get distinct rt_invoice from shipment_lines (only non-cancelled, non-deleted)
            $shipmentInvoices = DB::table('shipment_lines')
                ->join('customer_order_lines', 'shipment_lines.customer_order_lines_id', '=', 'customer_order_lines.id')
                ->join('customer_orders', 'customer_order_lines.customer_orders_id', '=', 'customer_orders.id')
                ->whereNotNull('shipment_lines.rt_invoice')
                ->where('shipment_lines.rt_invoice', '!=', '')
                ->whereNull('shipment_lines.deleted_at')
                ->whereNull('customer_order_lines.deleted_at')
                ->whereNull('customer_orders.deleted_at')
                ->where('customer_orders.cancelled', false)
                ->where('customer_order_lines.cancelled', false)
                ->distinct()
                ->count('shipment_lines.rt_invoice');

            return $samplesInvoices + $shipmentInvoices;
        });
    }

    public function render()
    {
        return view('livewire.dashboard.dashboard-metrics');
    }
}
