<?php

namespace App\Livewire;

use Livewire\Component;
use App\Models\Item;
use App\Models\Colourway;
use Illuminate\Support\Facades\Cookie;

class Basket extends Component
{
    public $basketItems = [];
    public $subtotal = 0;
    public $shipping = 5.00;
    public $total = 0;

    public function mount()
    {
        $this->loadBasket();
    }

    public function loadBasket()
    {
        $basket = json_decode(request()->cookie('basket', '[]'), true);
        $this->basketItems = [];
        $this->subtotal = 0;

        foreach ($basket as $item) {
            $product = Item::with(['colourways' => function($q) {
                $q->orderBy('sort_order');
            }])->find($item['product_id']);
            if ($product) {
                $colourway = null;
                if (isset($item['colourway_id'])) {
                    $colourway = Colourway::find($item['colourway_id']);
                }

                $itemTotal = $product->price * $item['quantity'];
                $this->subtotal += $itemTotal;

                $this->basketItems[] = [
                    'key' => $item['key'],
                    'product' => $product,
                    'colourway' => $colourway,
                    'quantity' => $item['quantity'],
                    'item_total' => $itemTotal,
                ];
            }
        }

        // Free shipping over £50
        if ($this->subtotal >= 50) {
            $this->shipping = 0;
        }

        $this->total = $this->subtotal + $this->shipping;
    }

    public function updateQuantity($key, $newQuantity)
    {
        if ($newQuantity < 1) {
            return;
        }

        $basket = json_decode(request()->cookie('basket', '[]'), true);
        
        foreach ($basket as &$item) {
            if ($item['key'] === $key) {
                $item['quantity'] = $newQuantity;
                break;
            }
        }

        Cookie::queue('basket', json_encode($basket), 60 * 24 * 30);
        $this->loadBasket();
        $this->dispatch('basket-updated');
    }

    public function removeItem($key)
    {
        $basket = json_decode(request()->cookie('basket', '[]'), true);
        $basket = array_filter($basket, fn($item) => $item['key'] !== $key);
        $basket = array_values($basket); // Re-index array

        Cookie::queue('basket', json_encode($basket), 60 * 24 * 30);
        $this->loadBasket();
        $this->dispatch('basket-updated');
    }

    public function clearBasket()
    {
        Cookie::queue('basket', json_encode([]), 60 * 24 * 30);
        $this->loadBasket();
        $this->dispatch('basket-updated');
    }

    public function render()
    {
        return view('livewire.basket')->layout('layouts.shop');
    }
}
