<?php

namespace App\Livewire;

use Livewire\Component;
use App\Models\Item;
use App\Models\Colourway;
use Illuminate\Support\Facades\Cookie;

class Basket extends Component
{
    public $basket = [];
    public $subtotal = 0;
    public $tax = 0;
    public $total = 0;

    protected $listeners = ['basket-updated' => '$refresh'];

    public function mount()
    {
        $this->loadBasket();
        $this->calculateTotals();
    }

    public function loadBasket()
    {
        $basketCookie = Cookie::get('basket', '[]');
        $basketData = json_decode($basketCookie, true);
        
        $this->basket = [];
        
        foreach ($basketData as $item) {
            $product = Item::find($item['item_id']);
            $colourway = isset($item['colourway_id']) ? Colourway::find($item['colourway_id']) : null;
            
            if ($product) {
                $this->basket[] = [
                    'item_id' => $product->id,
                    'colourway_id' => $colourway?->id,
                    'name' => $product->name,
                    'colourway_name' => $colourway?->reference ?? 'Default',
                    'price' => $product->price,
                    'quantity' => $item['quantity'],
                    'image' => $product->primary_image,
                    'slug' => $product->slug,
                ];
            }
        }
    }

    public function updateQuantity($index, $quantity)
    {
        if ($quantity <= 0) {
            $this->removeItem($index);
            return;
        }

        $this->basket[$index]['quantity'] = $quantity;
        $this->saveBasket();
        $this->calculateTotals();
        $this->dispatch('basket-updated');
    }

    public function removeItem($index)
    {
        unset($this->basket[$index]);
        $this->basket = array_values($this->basket);
        $this->saveBasket();
        $this->calculateTotals();
        $this->dispatch('basket-updated');
    }

    public function clearBasket()
    {
        $this->basket = [];
        $this->saveBasket();
        $this->calculateTotals();
        $this->dispatch('basket-updated');
    }

    private function saveBasket()
    {
        $basketData = array_map(function ($item) {
            return [
                'item_id' => $item['item_id'],
                'colourway_id' => $item['colourway_id'],
                'quantity' => $item['quantity'],
            ];
        }, $this->basket);

        Cookie::queue('basket', json_encode($basketData), 60 * 24 * 30); // 30 days
    }

    private function calculateTotals()
    {
        $this->subtotal = 0;
        
        foreach ($this->basket as $item) {
            $this->subtotal += $item['price'] * $item['quantity'];
        }
        
        // UK VAT is 20%
        $this->tax = $this->subtotal * 0.20;
        $this->total = $this->subtotal + $this->tax;
    }

    public function render()
    {
        return view('livewire.basket')->layout('layouts.frontend');
    }
}
