<?php

namespace App\Http\Controllers\Api\V1;

use App\Http\Controllers\Controller;
use App\Models\LibraryCategory;
use App\Models\LibraryTrack;
use App\Models\CustomRequest;
use App\Models\SafetyEvent;
use App\Services\CustomSessionService;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Http\Response;
use Illuminate\Support\Facades\Auth;

class AdminController extends Controller
{
    public function __construct(
        private CustomSessionService $customSessionService
    ) {}

    public function __construct()
    {
        $this->middleware('auth:sanctum');
        $this->middleware(function ($request, $next) {
            if (!Auth::user()->isAdmin()) {
                return response()->json([
                    'error' => 'Unauthorized',
                    'message' => 'Admin access required.',
                ], Response::HTTP_FORBIDDEN);
            }
            return $next($request);
        });
    }

    public function categories(): JsonResponse
    {
        $categories = LibraryCategory::withCount('tracks')->orderBy('sort_order')->get();
        
        return response()->json([
            'data' => $categories->map(fn($category) => [
                'id' => $category->id,
                'name' => $category->name,
                'slug' => $category->slug,
                'description' => $category->description,
                'icon' => $category->icon,
                'color' => $category->color,
                'sortOrder' => $category->sort_order,
                'isActive' => $category->is_active,
                'tracksCount' => $category->tracks_count,
                'createdAt' => $category->created_at,
            ]),
        ]);
    }

    public function tracks(): JsonResponse
    {
        $tracks = LibraryTrack::with('category')->orderBy('created_at', 'desc')->paginate(20);
        
        return response()->json([
            'data' => $tracks->items(),
            'meta' => [
                'current_page' => $tracks->currentPage(),
                'last_page' => $tracks->lastPage(),
                'per_page' => $tracks->perPage(),
                'total' => $tracks->total(),
            ],
        ]);
    }

    public function pendingRequests(): JsonResponse
    {
        $requests = $this->customSessionService->getPendingRequests();
        
        return response()->json([
            'data' => $requests->map(fn($request) => [
                'id' => $request->id,
                'topic' => $request->topic,
                'description' => $request->description,
                'length' => $request->length,
                'voicePreference' => $request->voice_preference,
                'specialNotes' => $request->special_notes,
                'status' => $request->status,
                'user' => [
                    'id' => $request->user->id,
                    'name' => $request->user->name,
                    'email' => $request->user->email,
                ],
                'createdAt' => $request->created_at,
            ]),
        ]);
    }

    public function approveRequest(int $id): JsonResponse
    {
        $request = CustomRequest::findOrFail($id);
        
        if ($request->status !== 'pending') {
            return response()->json([
                'error' => 'Invalid request status',
                'message' => 'Only pending requests can be approved.',
            ], Response::HTTP_BAD_REQUEST);
        }
        
        $this->customSessionService->approveRequest($request);
        
        return response()->json([
            'message' => 'Request approved successfully',
            'data' => [
                'id' => $request->id,
                'status' => $request->status,
            ],
        ]);
    }

    public function rejectRequest(Request $request, int $id): JsonResponse
    {
        $customRequest = CustomRequest::findOrFail($id);
        
        $request->validate([
            'reason' => 'required|string|max:500',
        ]);
        
        if ($customRequest->status !== 'pending') {
            return response()->json([
                'error' => 'Invalid request status',
                'message' => 'Only pending requests can be rejected.',
            ], Response::HTTP_BAD_REQUEST);
        }
        
        $this->customSessionService->rejectRequest($customRequest, $request->reason);
        
        return response()->json([
            'message' => 'Request rejected successfully',
            'data' => [
                'id' => $customRequest->id,
                'status' => $customRequest->status,
                'adminNotes' => $customRequest->admin_notes,
            ],
        ]);
    }

    public function safetyEvents(): JsonResponse
    {
        $events = SafetyEvent::with('user')
            ->orderBy('created_at', 'desc')
            ->paginate(20);
        
        return response()->json([
            'data' => $events->items(),
            'meta' => [
                'current_page' => $events->currentPage(),
                'last_page' => $events->lastPage(),
                'per_page' => $events->perPage(),
                'total' => $events->total(),
            ],
        ]);
    }

    public function resolveSafetyEvent(Request $request, int $id): JsonResponse
    {
        $event = SafetyEvent::findOrFail($id);
        
        $request->validate([
            'notes' => 'sometimes|nullable|string|max:1000',
        ]);
        
        if ($event->isResolved()) {
            return response()->json([
                'error' => 'Event already resolved',
                'message' => 'This safety event has already been resolved.',
            ], Response::HTTP_BAD_REQUEST);
        }
        
        $event->markAsResolved($request->notes);
        
        return response()->json([
            'message' => 'Safety event resolved successfully',
            'data' => [
                'id' => $event->id,
                'status' => $event->status,
                'resolvedAt' => $event->resolved_at,
                'resolutionNotes' => $event->resolution_notes,
            ],
        ]);
    }
}
