<?php

namespace App\Http\Controllers\Api\V1;

use App\Http\Controllers\Controller;
use App\Models\LibraryCategory;
use App\Models\LibraryTrack;
use App\Models\SafetyEvent;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Str;

class AdminController extends Controller
{
    public function categories(): JsonResponse
    {
        $categories = LibraryCategory::withCount('tracks')
            ->orderBy('sort_order')
            ->orderBy('name')
            ->get();

        return response()->json([
            'categories' => $categories->map(function ($category) {
                return [
                    'id' => $category->id,
                    'name' => $category->name,
                    'slug' => $category->slug,
                    'description' => $category->description,
                    'icon' => $category->icon,
                    'color' => $category->color,
                    'sortOrder' => $category->sort_order,
                    'isActive' => $category->is_active,
                    'tracksCount' => $category->tracks_count,
                    'createdAt' => $category->created_at,
                    'updatedAt' => $category->updated_at,
                ];
            }),
        ]);
    }

    public function storeCategory(Request $request): JsonResponse
    {
        $request->validate([
            'name' => 'required|string|max:255|unique:library_categories,name',
            'description' => 'nullable|string|max:1000',
            'icon' => 'nullable|string|max:100',
            'color' => 'nullable|string|max:7|regex:/^#[0-9A-F]{6}$/i',
            'sortOrder' => 'nullable|integer|min:0',
            'isActive' => 'boolean',
        ]);

        $category = LibraryCategory::create([
            'name' => $request->name,
            'slug' => Str::slug($request->name),
            'description' => $request->description,
            'icon' => $request->icon,
            'color' => $request->color ?? '#4dbda0',
            'sort_order' => $request->sortOrder ?? 0,
            'is_active' => $request->isActive ?? true,
        ]);

        return response()->json([
            'message' => 'Category created successfully',
            'category' => [
                'id' => $category->id,
                'name' => $category->name,
                'slug' => $category->slug,
                'description' => $category->description,
                'icon' => $category->icon,
                'color' => $category->color,
                'sortOrder' => $category->sort_order,
                'isActive' => $category->is_active,
            ],
        ], 201);
    }

    public function updateCategory(Request $request, LibraryCategory $category): JsonResponse
    {
        $request->validate([
            'name' => 'required|string|max:255|unique:library_categories,name,' . $category->id,
            'description' => 'nullable|string|max:1000',
            'icon' => 'nullable|string|max:100',
            'color' => 'nullable|string|max:7|regex:/^#[0-9A-F]{6}$/i',
            'sortOrder' => 'nullable|integer|min:0',
            'isActive' => 'boolean',
        ]);

        $category->update([
            'name' => $request->name,
            'slug' => Str::slug($request->name),
            'description' => $request->description,
            'icon' => $request->icon,
            'color' => $request->color,
            'sort_order' => $request->sortOrder,
            'is_active' => $request->isActive,
        ]);

        return response()->json([
            'message' => 'Category updated successfully',
            'category' => [
                'id' => $category->id,
                'name' => $category->name,
                'slug' => $category->slug,
                'description' => $category->description,
                'icon' => $category->icon,
                'color' => $category->color,
                'sortOrder' => $category->sort_order,
                'isActive' => $category->is_active,
            ],
        ]);
    }

    public function destroyCategory(LibraryCategory $category): JsonResponse
    {
        if ($category->tracks()->count() > 0) {
            return response()->json([
                'message' => 'Cannot delete category with existing tracks',
            ], 422);
        }

        $category->delete();

        return response()->json([
            'message' => 'Category deleted successfully',
        ]);
    }

    public function tracks(): JsonResponse
    {
        $tracks = LibraryTrack::with('category')
            ->orderBy('title')
            ->get();

        return response()->json([
            'tracks' => $tracks->map(function ($track) {
                return [
                    'id' => $track->id,
                    'title' => $track->title,
                    'slug' => $track->slug,
                    'description' => $track->description,
                    'duration' => $track->duration,
                    'formattedDuration' => $track->formatted_duration,
                    'fileSize' => $track->file_size,
                    'formattedFileSize' => $track->formatted_file_size,
                    'voiceType' => $track->voice_type,
                    'difficultyLevel' => $track->difficulty_level,
                    'tags' => $track->tags,
                    'safetyNotes' => $track->safety_notes,
                    'isActive' => $track->is_active,
                    'playCount' => $track->play_count,
                    'rating' => $track->rating,
                    'category' => $track->category ? [
                        'id' => $track->category->id,
                        'name' => $track->category->name,
                        'slug' => $track->category->slug,
                        'color' => $track->category->color,
                    ] : null,
                    'createdAt' => $track->created_at,
                    'updatedAt' => $track->updated_at,
                ];
            }),
        ]);
    }

    public function storeTrack(Request $request): JsonResponse
    {
        $request->validate([
            'categoryId' => 'required|exists:library_categories,id',
            'title' => 'required|string|max:255|unique:library_tracks,title',
            'description' => 'required|string|max:1000',
            'transcript' => 'required|string',
            'duration' => 'required|integer|min:60|max:7200', // 1 min to 2 hours
            'filePath' => 'required|string|max:500',
            'fileSize' => 'required|integer|min:1',
            'voiceType' => 'required|string|in:male,female,neutral',
            'difficultyLevel' => 'required|string|in:beginner,intermediate,advanced',
            'tags' => 'nullable|array',
            'tags.*' => 'string|max:50',
            'safetyNotes' => 'nullable|array',
            'safetyNotes.*' => 'string|max:200',
            'isActive' => 'boolean',
        ]);

        $track = LibraryTrack::create([
            'category_id' => $request->categoryId,
            'title' => $request->title,
            'slug' => Str::slug($request->title),
            'description' => $request->description,
            'transcript' => $request->transcript,
            'duration' => $request->duration,
            'file_path' => $request->filePath,
            'file_size' => $request->fileSize,
            'voice_type' => $request->voiceType,
            'difficulty_level' => $request->difficultyLevel,
            'tags' => $request->tags ?? [],
            'safety_notes' => $request->safetyNotes ?? [],
            'is_active' => $request->isActive ?? true,
            'play_count' => 0,
            'rating' => 0.0,
        ]);

        return response()->json([
            'message' => 'Track created successfully',
            'track' => [
                'id' => $track->id,
                'title' => $track->title,
                'slug' => $track->slug,
                'description' => $track->description,
                'duration' => $track->duration,
                'formattedDuration' => $track->formatted_duration,
                'voiceType' => $track->voice_type,
                'difficultyLevel' => $track->difficulty_level,
                'tags' => $track->tags,
                'isActive' => $track->is_active,
            ],
        ], 201);
    }

    public function updateTrack(Request $request, LibraryTrack $track): JsonResponse
    {
        $request->validate([
            'categoryId' => 'required|exists:library_categories,id',
            'title' => 'required|string|max:255|unique:library_tracks,title,' . $track->id,
            'description' => 'required|string|max:1000',
            'transcript' => 'required|string',
            'duration' => 'required|integer|min:60|max:7200',
            'filePath' => 'required|string|max:500',
            'fileSize' => 'required|integer|min:1',
            'voiceType' => 'required|string|in:male,female,neutral',
            'difficultyLevel' => 'required|string|in:beginner,intermediate,advanced',
            'tags' => 'nullable|array',
            'tags.*' => 'string|max:50',
            'safetyNotes' => 'nullable|array',
            'safetyNotes.*' => 'string|max:200',
            'isActive' => 'boolean',
        ]);

        $track->update([
            'category_id' => $request->categoryId,
            'title' => $request->title,
            'slug' => Str::slug($request->title),
            'description' => $request->description,
            'transcript' => $request->transcript,
            'duration' => $request->duration,
            'file_path' => $request->filePath,
            'file_size' => $request->fileSize,
            'voice_type' => $request->voiceType,
            'difficulty_level' => $request->difficultyLevel,
            'tags' => $request->tags ?? [],
            'safety_notes' => $request->safetyNotes ?? [],
            'is_active' => $request->isActive,
        ]);

        return response()->json([
            'message' => 'Track updated successfully',
            'track' => [
                'id' => $track->id,
                'title' => $track->title,
                'slug' => $track->slug,
                'description' => $track->description,
                'duration' => $track->duration,
                'formattedDuration' => $track->formatted_duration,
                'voiceType' => $track->voice_type,
                'difficultyLevel' => $track->difficulty_level,
                'tags' => $track->tags,
                'isActive' => $track->is_active,
            ],
        ]);
    }

    public function destroyTrack(LibraryTrack $track): JsonResponse
    {
        $track->delete();

        return response()->json([
            'message' => 'Track deleted successfully',
        ]);
    }

    public function safetyEvents(): JsonResponse
    {
        $events = SafetyEvent::with(['user', 'resolvedBy'])
            ->orderBy('created_at', 'desc')
            ->get();

        return response()->json([
            'events' => $events->map(function ($event) {
                return [
                    'id' => $event->id,
                    'eventType' => $event->event_type,
                    'severity' => $event->severity,
                    'description' => $event->description,
                    'metadata' => $event->metadata,
                    'status' => $event->status,
                    'resolutionNotes' => $event->resolution_notes,
                    'resolvedAt' => $event->resolved_at,
                    'user' => $event->user ? [
                        'id' => $event->user->id,
                        'name' => $event->user->name,
                        'email' => $event->user->email,
                    ] : null,
                    'resolvedBy' => $event->resolvedBy ? [
                        'id' => $event->resolvedBy->id,
                        'name' => $event->resolvedBy->name,
                    ] : null,
                    'createdAt' => $event->created_at,
                    'updatedAt' => $event->updated_at,
                ];
            }),
        ]);
    }
}
