<?php

namespace Tests\Unit;

use Tests\TestCase;
use App\Models\Price;
use App\Models\PriceResolution;
use App\Models\Colourways;
use App\Models\Customer;
use App\Models\Suppliers;
use App\Models\StyleVersions;
use App\Models\Styles;
use App\Models\Designs;
use App\Models\Seasons;
use App\Services\PriceResolutionService;
use Illuminate\Foundation\Testing\RefreshDatabase;

class PriceResolutionServiceTest extends TestCase
{
    use RefreshDatabase;

    protected PriceResolutionService $service;

    protected function setUp(): void
    {
        parent::setUp();
        $this->service = app(PriceResolutionService::class);
    }

    /**
     * Test that resolve method returns existing fresh resolution
     */
    public function test_resolve_returns_existing_fresh_resolution()
    {
        // Create test data
        $season = Seasons::factory()->create();
        $customer = Customer::factory()->create();
        $supplier = Suppliers::factory()->create();
        $design = Designs::factory()->create();
        $style = Styles::factory()->create([
            'customers_id' => $customer->id,
            'designs_id' => $design->id,
            'seasons_id' => $season->id,
        ]);
        $styleVersion = StyleVersions::factory()->create([
            'styles_id' => $style->id,
            'factory_id' => $supplier->id,
        ]);
        $colourway = Colourways::factory()->create([
            'style_versions_id' => $styleVersion->id,
        ]);
        $price = Price::factory()->create([
            'style_versions_id' => $styleVersion->id,
            'colourways_id' => $colourway->id,
        ]);

        // Create a fresh resolution manually
        $existingResolution = PriceResolution::create([
            'style_versions_id' => $styleVersion->id,
            'colourways_id' => $colourway->id,
            'sizes_id' => null,
            'phase_id' => null,
            'season_id' => $season->id,
            'customer_id' => $customer->id,
            'supplier_id' => $supplier->id,
            'style_id' => $style->id,
            'design_id' => $design->id,
            'price_id' => $price->id,
            'cmt' => 10.50,
            'quote' => 15.75,
            'fresh_at' => now(),
        ]);

        // Call resolve method
        $result = $this->service->resolve(
            $styleVersion->id,
            $colourway->id,
            null,
            null,
            $season->id
        );

        // Should return the existing resolution
        $this->assertNotNull($result);
        $this->assertEquals($existingResolution->id, $result->id);
        $this->assertNotNull($result->fresh_at);
    }

    /**
     * Test that resolve method computes new resolution when none exists
     */
    public function test_resolve_computes_new_resolution_when_none_exists()
    {
        // Create test data
        $season = Seasons::factory()->create();
        $customer = Customer::factory()->create();
        $supplier = Suppliers::factory()->create();
        $design = Designs::factory()->create();
        $style = Styles::factory()->create([
            'customers_id' => $customer->id,
            'designs_id' => $design->id,
            'seasons_id' => $season->id,
        ]);
        $styleVersion = StyleVersions::factory()->create([
            'styles_id' => $style->id,
            'factory_id' => $supplier->id,
        ]);
        $colourway = Colourways::factory()->create([
            'style_versions_id' => $styleVersion->id,
        ]);
        $price = Price::factory()->create([
            'style_versions_id' => $styleVersion->id,
            'colourways_id' => $colourway->id,
        ]);

        // Call resolve method
        $result = $this->service->resolve(
            $styleVersion->id,
            $colourway->id,
            null,
            null,
            $season->id
        );

        // Should create a new resolution
        $this->assertNotNull($result);
        $this->assertNotNull($result->fresh_at);
        $this->assertEquals($styleVersion->id, $result->style_versions_id);
        $this->assertEquals($colourway->id, $result->colourways_id);
        $this->assertEquals($season->id, $result->season_id);
    }

    /**
     * Test that invalidateByColourway marks resolutions as stale
     */
    public function test_invalidate_by_colourway_marks_resolutions_stale()
    {
        // Create test data
        $season = Seasons::factory()->create();
        $customer = Customer::factory()->create();
        $supplier = Suppliers::factory()->create();
        $design = Designs::factory()->create();
        $style = Styles::factory()->create([
            'customers_id' => $customer->id,
            'designs_id' => $design->id,
            'seasons_id' => $season->id,
        ]);
        $styleVersion = StyleVersions::factory()->create([
            'styles_id' => $style->id,
            'factory_id' => $supplier->id,
        ]);
        $colourway = Colourways::factory()->create([
            'style_versions_id' => $styleVersion->id,
        ]);
        $price = Price::factory()->create([
            'style_versions_id' => $styleVersion->id,
            'colourways_id' => $colourway->id,
        ]);

        // Create resolutions
        $resolution1 = PriceResolution::create([
            'style_versions_id' => $styleVersion->id,
            'colourways_id' => $colourway->id,
            'sizes_id' => null,
            'phase_id' => null,
            'season_id' => $season->id,
            'customer_id' => $customer->id,
            'supplier_id' => $supplier->id,
            'style_id' => $style->id,
            'design_id' => $design->id,
            'price_id' => $price->id,
            'cmt' => 10.50,
            'quote' => 15.75,
            'fresh_at' => now(),
        ]);

        $resolution2 = PriceResolution::create([
            'style_versions_id' => $styleVersion->id,
            'colourways_id' => $colourway->id,
            'sizes_id' => 1,
            'phase_id' => null,
            'season_id' => $season->id,
            'customer_id' => $customer->id,
            'supplier_id' => $supplier->id,
            'style_id' => $style->id,
            'design_id' => $design->id,
            'price_id' => $price->id,
            'cmt' => 10.50,
            'quote' => 15.75,
            'fresh_at' => now(),
        ]);

        // Invalidate by colourway
        $invalidatedCount = $this->service->invalidateByColourway($colourway->id);

        // Check that resolutions are marked as stale
        $this->assertEquals(2, $invalidatedCount);
        
        $resolution1->refresh();
        $resolution2->refresh();
        
        $this->assertNull($resolution1->fresh_at);
        $this->assertNull($resolution2->fresh_at);
    }

    /**
     * Test that invalidateByCustomer marks resolutions as stale
     */
    public function test_invalidate_by_customer_marks_resolutions_stale()
    {
        // Create test data
        $season = Seasons::factory()->create();
        $customer = Customer::factory()->create();
        $supplier = Suppliers::factory()->create();
        $design = Designs::factory()->create();
        $style = Styles::factory()->create([
            'customers_id' => $customer->id,
            'designs_id' => $design->id,
            'seasons_id' => $season->id,
        ]);
        $styleVersion = StyleVersions::factory()->create([
            'styles_id' => $style->id,
            'factory_id' => $supplier->id,
        ]);
        $colourway = Colourways::factory()->create([
            'style_versions_id' => $styleVersion->id,
        ]);
        $price = Price::factory()->create([
            'style_versions_id' => $styleVersion->id,
            'colourways_id' => $colourway->id,
        ]);

        // Create resolution
        $resolution = PriceResolution::create([
            'style_versions_id' => $styleVersion->id,
            'colourways_id' => $colourway->id,
            'sizes_id' => null,
            'phase_id' => null,
            'season_id' => $season->id,
            'customer_id' => $customer->id,
            'supplier_id' => $supplier->id,
            'style_id' => $style->id,
            'design_id' => $design->id,
            'price_id' => $price->id,
            'cmt' => 10.50,
            'quote' => 15.75,
            'fresh_at' => now(),
        ]);

        // Invalidate by customer
        $invalidatedCount = $this->service->invalidateByCustomer($customer->id);

        // Check that resolution is marked as stale
        $this->assertEquals(1, $invalidatedCount);
        
        $resolution->refresh();
        $this->assertNull($resolution->fresh_at);
    }

    /**
     * Test that invalidateBySupplier marks resolutions as stale
     */
    public function test_invalidate_by_supplier_marks_resolutions_stale()
    {
        // Create test data
        $season = Seasons::factory()->create();
        $customer = Customer::factory()->create();
        $supplier = Suppliers::factory()->create();
        $design = Designs::factory()->create();
        $style = Styles::factory()->create([
            'customers_id' => $customer->id,
            'designs_id' => $design->id,
            'seasons_id' => $season->id,
        ]);
        $styleVersion = StyleVersions::factory()->create([
            'styles_id' => $style->id,
            'factory_id' => $supplier->id,
        ]);
        $colourway = Colourways::factory()->create([
            'style_versions_id' => $styleVersion->id,
        ]);
        $price = Price::factory()->create([
            'style_versions_id' => $styleVersion->id,
            'colourways_id' => $colourway->id,
        ]);

        // Create resolution
        $resolution = PriceResolution::create([
            'style_versions_id' => $styleVersion->id,
            'colourways_id' => $colourway->id,
            'sizes_id' => null,
            'phase_id' => null,
            'season_id' => $season->id,
            'customer_id' => $customer->id,
            'supplier_id' => $supplier->id,
            'style_id' => $style->id,
            'design_id' => $design->id,
            'price_id' => $price->id,
            'cmt' => 10.50,
            'quote' => 15.75,
            'fresh_at' => now(),
        ]);

        // Invalidate by supplier
        $invalidatedCount = $this->service->invalidateBySupplier($supplier->id);

        // Check that resolution is marked as stale
        $this->assertEquals(1, $invalidatedCount);
        
        $resolution->refresh();
        $this->assertNull($resolution->fresh_at);
    }

    /**
     * Test that invalidateByPrice marks resolutions as stale
     */
    public function test_invalidate_by_price_marks_resolutions_stale()
    {
        // Create test data
        $season = Seasons::factory()->create();
        $customer = Customer::factory()->create();
        $supplier = Suppliers::factory()->create();
        $design = Designs::factory()->create();
        $style = Styles::factory()->create([
            'customers_id' => $customer->id,
            'designs_id' => $design->id,
            'seasons_id' => $season->id,
        ]);
        $styleVersion = StyleVersions::factory()->create([
            'styles_id' => $style->id,
            'factory_id' => $supplier->id,
        ]);
        $colourway = Colourways::factory()->create([
            'style_versions_id' => $styleVersion->id,
        ]);
        $price = Price::factory()->create([
            'style_versions_id' => $styleVersion->id,
            'colourways_id' => $colourway->id,
        ]);

        // Create resolution
        $resolution = PriceResolution::create([
            'style_versions_id' => $styleVersion->id,
            'colourways_id' => $colourway->id,
            'sizes_id' => null,
            'phase_id' => null,
            'season_id' => $season->id,
            'customer_id' => $customer->id,
            'supplier_id' => $supplier->id,
            'style_id' => $style->id,
            'design_id' => $design->id,
            'price_id' => $price->id,
            'cmt' => 10.50,
            'quote' => 15.75,
            'fresh_at' => now(),
        ]);

        // Invalidate by price
        $invalidatedCount = $this->service->invalidateByPrice($price->id);

        // Check that resolution is marked as stale
        $this->assertEquals(1, $invalidatedCount);
        
        $resolution->refresh();
        $this->assertNull($resolution->fresh_at);
    }

    /**
     * Test that getCacheStats returns correct statistics
     */
    public function test_get_cache_stats_returns_correct_statistics()
    {
        // Create test data
        $season = Seasons::factory()->create();
        $customer = Customer::factory()->create();
        $supplier = Suppliers::factory()->create();
        $design = Designs::factory()->create();
        $style = Styles::factory()->create([
            'customers_id' => $customer->id,
            'designs_id' => $design->id,
            'seasons_id' => $season->id,
        ]);
        $styleVersion = StyleVersions::factory()->create([
            'styles_id' => $style->id,
            'factory_id' => $supplier->id,
        ]);
        $colourway = Colourways::factory()->create([
            'style_versions_id' => $styleVersion->id,
        ]);
        $price = Price::factory()->create([
            'style_versions_id' => $styleVersion->id,
            'colourways_id' => $colourway->id,
        ]);

        // Create fresh resolution
        PriceResolution::create([
            'style_versions_id' => $styleVersion->id,
            'colourways_id' => $colourway->id,
            'sizes_id' => null,
            'phase_id' => null,
            'season_id' => $season->id,
            'customer_id' => $customer->id,
            'supplier_id' => $supplier->id,
            'style_id' => $style->id,
            'design_id' => $design->id,
            'price_id' => $price->id,
            'cmt' => 10.50,
            'quote' => 15.75,
            'fresh_at' => now(),
        ]);

        // Create stale resolution
        PriceResolution::create([
            'style_versions_id' => $styleVersion->id,
            'colourways_id' => $colourway->id,
            'sizes_id' => 1,
            'phase_id' => null,
            'season_id' => $season->id,
            'customer_id' => $customer->id,
            'supplier_id' => $supplier->id,
            'style_id' => $style->id,
            'design_id' => $design->id,
            'price_id' => $price->id,
            'cmt' => 10.50,
            'quote' => 15.75,
            'fresh_at' => now()->subDays(2), // Stale
        ]);

        // Create null fresh_at resolution
        PriceResolution::create([
            'style_versions_id' => $styleVersion->id,
            'colourways_id' => $colourway->id,
            'sizes_id' => 2,
            'phase_id' => null,
            'season_id' => $season->id,
            'customer_id' => $customer->id,
            'supplier_id' => $supplier->id,
            'style_id' => $style->id,
            'design_id' => $design->id,
            'price_id' => $price->id,
            'cmt' => 10.50,
            'quote' => 15.75,
            'fresh_at' => null, // Stale
        ]);

        // Get cache stats
        $stats = $this->service->getCacheStats();

        // Check statistics
        $this->assertEquals(3, $stats['total']);
        $this->assertEquals(1, $stats['fresh']);
        $this->assertEquals(2, $stats['stale']);
        $this->assertEquals(33.33, $stats['fresh_percentage']);
    }

    /**
     * Test that cleanupStaleResolutions removes old resolutions
     */
    public function test_cleanup_stale_resolutions_removes_old_resolutions()
    {
        // Create test data
        $season = Seasons::factory()->create();
        $customer = Customer::factory()->create();
        $supplier = Suppliers::factory()->create();
        $design = Designs::factory()->create();
        $style = Styles::factory()->create([
            'customers_id' => $customer->id,
            'designs_id' => $design->id,
            'seasons_id' => $season->id,
        ]);
        $styleVersion = StyleVersions::factory()->create([
            'styles_id' => $style->id,
            'factory_id' => $supplier->id,
        ]);
        $colourway = Colourways::factory()->create([
            'style_versions_id' => $styleVersion->id,
        ]);
        $price = Price::factory()->create([
            'style_versions_id' => $styleVersion->id,
            'colourways_id' => $colourway->id,
        ]);

        // Create old resolution
        PriceResolution::create([
            'style_versions_id' => $styleVersion->id,
            'colourways_id' => $colourway->id,
            'sizes_id' => null,
            'phase_id' => null,
            'season_id' => $season->id,
            'customer_id' => $customer->id,
            'supplier_id' => $supplier->id,
            'style_id' => $style->id,
            'design_id' => $design->id,
            'price_id' => $price->id,
            'cmt' => 10.50,
            'quote' => 15.75,
            'fresh_at' => now()->subDays(10), // Very old
        ]);

        // Create recent resolution
        PriceResolution::create([
            'style_versions_id' => $styleVersion->id,
            'colourways_id' => $colourway->id,
            'sizes_id' => 1,
            'phase_id' => null,
            'season_id' => $season->id,
            'customer_id' => $customer->id,
            'supplier_id' => $supplier->id,
            'style_id' => $style->id,
            'design_id' => $design->id,
            'price_id' => $price->id,
            'cmt' => 10.50,
            'quote' => 15.75,
            'fresh_at' => now()->subDays(1), // Recent
        ]);

        // Cleanup stale resolutions
        $deletedCount = $this->service->cleanupStaleResolutions(7);

        // Check that only old resolution was deleted
        $this->assertEquals(1, $deletedCount);
        $this->assertEquals(1, PriceResolution::count());
    }
}
