<?php

namespace Tests\Feature;

use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Support\Facades\Hash;
use Tests\TestCase;

class BootstrapTest extends TestCase
{
    use RefreshDatabase;

    protected function setUp(): void
    {
        parent::setUp();
        
        // Create test user
        User::create([
            'name' => 'Test User',
            'email' => 'test@example.com',
            'password' => Hash::make('password'),
            'role' => 'admin',
        ]);
    }

    /** @test */
    public function bootstrap_css_is_loaded()
    {
        $response = $this->get('/');
        
        $response->assertStatus(200);
        $response->assertSee('build/assets/app-');
    }

    /** @test */
    public function bootstrap_js_is_loaded()
    {
        $response = $this->get('/');
        
        $response->assertStatus(200);
        $response->assertSee('build/assets/app-');
    }

    /** @test */
    public function layout_uses_bootstrap_classes()
    {
        $response = $this->get('/');
        
        $response->assertStatus(200);
        $response->assertSee('container');
        $response->assertSee('navbar');
        $response->assertSee('card');
    }

    /** @test */
    public function login_page_uses_bootstrap_styling()
    {
        $response = $this->get('/login');
        
        $response->assertStatus(200);
        $response->assertSee('container');
        $response->assertSee('card');
        $response->assertSee('form-control');
        $response->assertSee('btn');
    }

    /** @test */
    public function register_page_uses_bootstrap_styling()
    {
        $response = $this->get('/register');
        
        $response->assertStatus(200);
        $response->assertSee('container');
        $response->assertSee('card');
        $response->assertSee('form-control');
        $response->assertSee('btn');
    }

    /** @test */
    public function navbar_has_bootstrap_classes()
    {
        $response = $this->get('/');
        
        $response->assertStatus(200);
        $response->assertSee('navbar-nav');
        $response->assertSee('nav-link');
    }

    /** @test */
    public function welcome_component_uses_bootstrap_classes()
    {
        $response = $this->get('/');
        
        $response->assertStatus(200);
        $response->assertSee('container');
        $response->assertSee('row');
        $response->assertSee('col-md-8');
        $response->assertSee('card');
        $response->assertSee('card-header');
        $response->assertSee('card-body');
    }

    /** @test */
    public function authenticated_user_sees_bootstrap_styled_content()
    {
        $user = User::where('email', 'test@example.com')->first();
        
        $response = $this->actingAs($user)->get('/home');
        
        $response->assertStatus(200);
        $response->assertSee('container');
        $response->assertSee('card');
        $response->assertSee('alert');
    }

    /** @test */
    public function forms_have_bootstrap_styling()
    {
        $response = $this->get('/login');
        
        $response->assertStatus(200);
        $response->assertSee('form-group');
        $response->assertSee('form-control');
        $response->assertSee('btn-primary');
    }

    /** @test */
    public function buttons_have_bootstrap_classes()
    {
        $response = $this->get('/login');
        
        $response->assertStatus(200);
        $response->assertSee('btn');
        $response->assertSee('btn-primary');
    }

    /** @test */
    public function alerts_have_bootstrap_classes()
    {
        $user = User::where('email', 'test@example.com')->first();
        
        $response = $this->actingAs($user)->get('/home');
        
        $response->assertStatus(200);
        $response->assertSee('alert');
        $response->assertSee('alert-success');
    }

    /** @test */
    public function list_groups_have_bootstrap_classes()
    {
        $response = $this->get('/');
        
        $response->assertStatus(200);
        $response->assertSee('list-group');
        $response->assertSee('list-group-item');
    }

    /** @test */
    public function responsive_classes_are_present()
    {
        $response = $this->get('/');
        
        $response->assertStatus(200);
        $response->assertSee('col-md-8');
        $response->assertSee('justify-content-center');
    }

    /** @test */
    public function bootstrap_icons_are_available()
    {
        $response = $this->get('/');
        
        $response->assertStatus(200);
        $response->assertSee('✅'); // Checkmark emoji used in features list
    }

    /** @test */
    public function navigation_has_bootstrap_styling()
    {
        $response = $this->get('/');
        
        $response->assertStatus(200);
        $response->assertSee('navbar-brand');
        $response->assertSee('navbar-toggler');
    }

    /** @test */
    public function admin_navigation_has_bootstrap_styling()
    {
        $user = User::where('email', 'test@example.com')->first();
        
        $response = $this->actingAs($user)->get('/home');
        
        $response->assertStatus(200);
        $response->assertSee('nav-item');
        $response->assertSee('nav-link');
    }

    /** @test */
    public function bootstrap_assets_are_accessible()
    {
        $response = $this->get('/');
        
        $response->assertStatus(200);
        
        // Extract the asset URLs from the response
        preg_match('/href="([^"]*\.css)"/', $response->getContent(), $cssMatches);
        preg_match('/src="([^"]*\.js)"/', $response->getContent(), $jsMatches);
        
        if (!empty($cssMatches[1])) {
            $cssResponse = $this->get($cssMatches[1]);
            $cssResponse->assertStatus(200);
        }
        
        if (!empty($jsMatches[1])) {
            $jsResponse = $this->get($jsMatches[1]);
            $jsResponse->assertStatus(200);
        }
    }

    /** @test */
    public function bootstrap_forms_are_properly_structured()
    {
        $response = $this->get('/login');
        
        $response->assertStatus(200);
        $response->assertSee('method="POST"');
        $response->assertSee('action="/login"');
        $response->assertSee('name="email"');
        $response->assertSee('name="password"');
        $response->assertSee('type="email"');
        $response->assertSee('type="password"');
    }

    /** @test */
    public function bootstrap_cards_are_properly_structured()
    {
        $response = $this->get('/');
        
        $response->assertStatus(200);
        $response->assertSee('card-header');
        $response->assertSee('card-body');
    }
}