<?php

namespace App\Models;

use App\Models\BaseModel;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\SoftDeletes;

class PriceResolution extends BaseModel
{
    use HasFactory;
    use SoftDeletes;

    protected $fillable = [
        'style_versions_id',
        'colourways_id',
        'sizes_id',
        'phase_id',
        'season_id',
        'customer_id',
        'supplier_id',
        'style_id',
        'design_id',
        'price_id',
        'cmt',
        'cmt_base',
        'quote',
        'quote_base',
        'discount_price',
        'discount_price_base',
        'gmt_trans_base',
        'subtotal',
        'subtotal_base',
        'yarn_value_euro',
        'cmt_status',
        'quote_status',
        'margin',
        'cust_currency',
        'fact_currency',
        'fresh_at',
    ];

    protected $casts = [
        'cmt' => 'decimal:4',
        'cmt_base' => 'decimal:4',
        'quote' => 'decimal:4',
        'quote_base' => 'decimal:4',
        'discount_price' => 'decimal:4',
        'discount_price_base' => 'decimal:4',
        'gmt_trans_base' => 'decimal:4',
        'subtotal' => 'decimal:4',
        'subtotal_base' => 'decimal:4',
        'yarn_value_euro' => 'decimal:4',
        'margin' => 'decimal:2',
        'fresh_at' => 'datetime',
        'created_at' => 'datetime',
        'updated_at' => 'datetime',
        'deleted_at' => 'datetime',
    ];

    /**
     * Get the style version that owns this price resolution.
     */
    public function styleVersion(): BelongsTo
    {
        return $this->belongsTo(StyleVersions::class, 'style_versions_id');
    }

    /**
     * Get the colourway that owns this price resolution.
     */
    public function colourway(): BelongsTo
    {
        return $this->belongsTo(Colourways::class, 'colourways_id');
    }

    /**
     * Get the size that owns this price resolution.
     */
    public function size(): BelongsTo
    {
        return $this->belongsTo(Sizes::class, 'sizes_id');
    }

    /**
     * Get the phase that owns this price resolution.
     */
    public function phase(): BelongsTo
    {
        return $this->belongsTo(Phase::class, 'phase_id');
    }

    /**
     * Get the season that owns this price resolution.
     */
    public function season(): BelongsTo
    {
        return $this->belongsTo(Seasons::class, 'season_id');
    }

    /**
     * Get the customer that owns this price resolution.
     */
    public function customer(): BelongsTo
    {
        return $this->belongsTo(Customers::class, 'customer_id');
    }

    /**
     * Get the supplier (factory) that owns this price resolution.
     */
    public function supplier(): BelongsTo
    {
        return $this->belongsTo(Suppliers::class, 'supplier_id');
    }

    /**
     * Get the style that owns this price resolution.
     */
    public function style(): BelongsTo
    {
        return $this->belongsTo(Styles::class, 'style_id');
    }

    /**
     * Get the design that owns this price resolution.
     */
    public function design(): BelongsTo
    {
        return $this->belongsTo(Designs::class, 'design_id');
    }

    /**
     * Get the price that this resolution is based on.
     */
    public function price(): BelongsTo
    {
        return $this->belongsTo(Price::class, 'price_id');
    }

    /**
     * Check if this resolution is fresh (recently computed).
     */
    public function isFresh(): bool
    {
        return $this->fresh_at && $this->fresh_at->isAfter(now()->subHours(24));
    }

    /**
     * Mark this resolution as fresh.
     */
    public function markAsFresh(): void
    {
        $this->update(['fresh_at' => now()]);
    }

    /**
     * Scope to get fresh resolutions only.
     */
    public function scopeFresh($query)
    {
        return $query->where('fresh_at', '>', now()->subHours(24));
    }

    /**
     * Scope to get stale resolutions only.
     */
    public function scopeStale($query)
    {
        return $query->where(function ($q) {
            $q->whereNull('fresh_at')
              ->orWhere('fresh_at', '<=', now()->subHours(24));
        });
    }
}
