<?php

namespace App\Livewire;

use Livewire\Component;
use Livewire\WithPagination;
use App\Models\Item;
use App\Models\Season;
use App\Models\Category;

class ShopAll extends Component
{
    use WithPagination;

    public $selectedSeason = null;
    public $selectedCategory = null;
    public $sortBy = 'latest';
    public $search = '';

    protected $queryString = ['selectedSeason', 'selectedCategory', 'sortBy', 'search'];

    public function updatedSelectedSeason()
    {
        $this->resetPage();
    }

    public function updatedSelectedCategory()
    {
        $this->resetPage();
    }

    public function updatedSortBy()
    {
        $this->resetPage();
    }

    public function updatedSearch()
    {
        $this->resetPage();
    }

    public function clearFilters()
    {
        $this->selectedSeason = null;
        $this->selectedCategory = null;
        $this->search = '';
        $this->resetPage();
    }

    public function render()
    {
        $query = Item::with(['season', 'category']);

        if ($this->selectedSeason) {
            $query->whereHas('season', function($q) {
                $q->where('slug', $this->selectedSeason);
            });
        }

        if ($this->selectedCategory) {
            $query->whereHas('category', function($q) {
                $q->where('slug', $this->selectedCategory);
            });
        }

        if ($this->search) {
            $query->where(function($q) {
                $q->where('name', 'like', '%' . $this->search . '%')
                  ->orWhere('description', 'like', '%' . $this->search . '%')
                  ->orWhere('sku', 'like', '%' . $this->search . '%');
            });
        }

        switch ($this->sortBy) {
            case 'price_asc':
                $query->orderBy('price', 'asc');
                break;
            case 'price_desc':
                $query->orderBy('price', 'desc');
                break;
            case 'name':
                $query->orderBy('name', 'asc');
                break;
            default:
                $query->latest();
        }

        $products = $query->paginate(12);

        $seasons = Season::orderBy('sort_order')->get();
        $categories = Category::orderBy('sort_order')->get();

        return view('livewire.shop-all', [
            'products' => $products,
            'seasons' => $seasons,
            'categories' => $categories,
        ])->layout('layouts.shop');
    }
}

