<?php

namespace Tests\Feature;

use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Support\Facades\Hash;
use Tests\TestCase;

class UserRoleTest extends TestCase
{
    use RefreshDatabase;

    protected function setUp(): void
    {
        parent::setUp();
        
        // Create test users with different roles
        User::create([
            'name' => 'Admin User',
            'email' => 'admin@example.com',
            'password' => Hash::make('password'),
            'role' => 'admin',
        ]);

        User::create([
            'name' => 'Regular User',
            'email' => 'user@example.com',
            'password' => Hash::make('password'),
            'role' => 'user',
        ]);
    }

    /** @test */
    public function user_model_has_role_attribute()
    {
        $admin = User::where('email', 'admin@example.com')->first();
        $user = User::where('email', 'user@example.com')->first();

        $this->assertEquals('admin', $admin->role);
        $this->assertEquals('user', $user->role);
    }

    /** @test */
    public function user_model_can_check_if_admin()
    {
        $admin = User::where('email', 'admin@example.com')->first();
        $user = User::where('email', 'user@example.com')->first();

        $this->assertTrue($admin->isAdmin());
        $this->assertFalse($user->isAdmin());
    }

    /** @test */
    public function user_model_can_check_if_regular_user()
    {
        $admin = User::where('email', 'admin@example.com')->first();
        $user = User::where('email', 'user@example.com')->first();

        $this->assertFalse($admin->isUser());
        $this->assertTrue($user->isUser());
    }

    /** @test */
    public function new_user_registration_gets_default_user_role()
    {
        $userData = [
            'name' => 'New User',
            'email' => 'newuser@example.com',
            'password' => 'password',
            'password_confirmation' => 'password',
        ];

        $this->post('/register', $userData);

        $user = User::where('email', 'newuser@example.com')->first();
        $this->assertEquals('user', $user->role);
        $this->assertTrue($user->isUser());
        $this->assertFalse($user->isAdmin());
    }

    /** @test */
    public function admin_user_can_see_admin_navigation()
    {
        $admin = User::where('email', 'admin@example.com')->first();
        
        $response = $this->actingAs($admin)->get('/home');
        
        $response->assertStatus(200);
        $response->assertSee('Admin Panel');
    }

    /** @test */
    public function regular_user_cannot_see_admin_navigation()
    {
        $user = User::where('email', 'user@example.com')->first();
        
        $response = $this->actingAs($user)->get('/home');
        
        $response->assertStatus(200);
        $response->assertDontSee('Admin Panel');
    }

    /** @test */
    public function role_is_fillable_in_user_model()
    {
        $user = User::create([
            'name' => 'Test User',
            'email' => 'test@example.com',
            'password' => Hash::make('password'),
            'role' => 'admin',
        ]);

        $this->assertEquals('admin', $user->role);
    }

    /** @test */
    public function user_role_can_be_updated()
    {
        $user = User::where('email', 'user@example.com')->first();
        
        $user->update(['role' => 'admin']);
        
        $this->assertEquals('admin', $user->fresh()->role);
        $this->assertTrue($user->fresh()->isAdmin());
    }

    /** @test */
    public function admin_user_sees_admin_privileges_message()
    {
        $admin = User::where('email', 'admin@example.com')->first();
        
        $response = $this->actingAs($admin)->get('/home');
        
        $response->assertStatus(200);
        $response->assertSee('admin privileges');
    }

    /** @test */
    public function regular_user_sees_user_message()
    {
        $user = User::where('email', 'user@example.com')->first();
        
        $response = $this->actingAs($user)->get('/home');
        
        $response->assertStatus(200);
        $response->assertSee('browse and purchase');
    }

    /** @test */
    public function user_role_validation_works()
    {
        $user = User::where('email', 'user@example.com')->first();
        
        // Test with invalid role
        $user->role = 'invalid_role';
        $this->assertFalse($user->isAdmin());
        $this->assertFalse($user->isUser());
    }

    /** @test */
    public function database_has_role_column()
    {
        $this->assertDatabaseHas('users', [
            'email' => 'admin@example.com',
            'role' => 'admin',
        ]);

        $this->assertDatabaseHas('users', [
            'email' => 'user@example.com',
            'role' => 'user',
        ]);
    }

    /** @test */
    public function user_can_have_null_role_and_defaults_to_user()
    {
        // This test ensures the database default works
        $user = User::create([
            'name' => 'No Role User',
            'email' => 'norole@example.com',
            'password' => Hash::make('password'),
            // role not specified, should default to 'user'
        ]);

        $this->assertEquals('user', $user->role);
        $this->assertTrue($user->isUser());
        $this->assertFalse($user->isAdmin());
    }
}