<?php

namespace App\Livewire\Finance;

use Livewire\Component;
use App\Models\Statement;
use App\Models\StatementTransaction;
use App\Models\Receipt;
use App\Models\ReceiptLine;
use App\Models\Account;
use App\Models\Department;
use Livewire\Attributes\On;
use App\Jobs\MatchReceipt;
use App\Jobs\ProcessReceipt;

class StatementReview extends Component
{
    public ?int $statementId = null;
    public bool $matchingRequested = false;
    public ?int $selectedReceiptId = null;
    public array $editReceipt = [];
    public array $transactions = [];
    public array $splits = [];
    public array $expanded = [];
    public array $selectedReceipts = [];
    public bool $showMatchButton = false;
    public bool $isMatching = false;
    public ?int $manualMatchReceiptId = null;
    public ?int $manualMatchTransactionId = null;
    public bool $showManualMatchModal = false;
    public string $manualMatchSearch = '';
    public string $manualMatchSortBy = 'date';
    public string $manualMatchSortOrder = 'desc';

    public function mount(?int $statementId = null)
    {
        // If no statement is selected, show all transactions by default
        // For now, always show all transactions to see all matches
        $this->statementId = null;
        $this->loadTransactions();
    }

    public function loadTransactions()
    {
        // Load all transactions from open statements
        $statements = Statement::where('status', 'open')->get();
        
        if ($statements->isEmpty()) {
            $this->transactions = [];
            return;
        }

        $this->transactions = StatementTransaction::with(['matches.receipt', 'statement'])
            ->whereIn('statement_id', $statements->pluck('id'))
            ->orderBy('transaction_date', 'desc')
            ->get()
            ->toArray();
    }

    public function render()
    {
        // Get unmatched receipts
        $unmatchedReceipts = Receipt::whereDoesntHave('matches')
            ->orderBy('created_at', 'desc')
            ->get();

        // Get matched receipts
        $matchedReceipts = Receipt::whereHas('matches')
            ->with('matches.statementTransaction')
            ->orderBy('created_at', 'desc')
            ->get();

        return view('livewire.finance.statement-review', [
            'unmatchedReceipts' => $unmatchedReceipts,
            'matchedReceipts' => $matchedReceipts,
            'unmatchedReceiptsCount' => $unmatchedReceipts->count(),
            'unmatchedTransactionsCount' => StatementTransaction::whereDoesntHave('matches')->count(),
        ]);
    }

    public function runMatching()
    {
        $this->matchingRequested = true;
        // Dispatch matching job for all unmatched receipts
        $unmatchedReceipts = Receipt::whereDoesntHave('matches')->get();
        foreach ($unmatchedReceipts as $receipt) {
            MatchReceipt::dispatch($receipt->id);
        }
        session()->flash('status', 'Matching process started for ' . $unmatchedReceipts->count() . ' receipts.');
    }

    public function matchReceipt(int $receiptId)
    {
        MatchReceipt::dispatch($receiptId);
        session()->flash('status', 'Matching process started for receipt #' . $receiptId);
    }

    public function reprocessReceipt(int $receiptId)
    {
        ProcessReceipt::dispatch($receiptId);
        session()->flash('status', 'Re-processing started for receipt #' . $receiptId);
    }

    public function selectReceipt(int $receiptId)
    {
        $this->selectedReceiptId = $receiptId;
        $receipt = Receipt::find($receiptId);
        if ($receipt) {
            $this->editReceipt = [
                'id' => $receipt->id,
                'merchant_name' => $receipt->merchant_name,
                'total_amount' => $receipt->total_amount,
                'receipt_date' => $receipt->receipt_date?->format('Y-m-d'),
                'currency' => $receipt->currency,
            ];
        }
    }

    public function closeModal()
    {
        $this->selectedReceiptId = null;
        $this->editReceipt = [];
    }

    public function saveReceiptEdits()
    {
        if (empty($this->editReceipt['id'])) {
            return;
        }

        $receipt = Receipt::find($this->editReceipt['id']);
        if (!$receipt) {
            return;
        }

        $receipt->update([
            'merchant_name' => $this->editReceipt['merchant_name'],
            'total_amount' => $this->editReceipt['total_amount'],
            'receipt_date' => $this->editReceipt['receipt_date'] ?: null,
            'currency' => $this->editReceipt['currency'],
        ]);

        session()->flash('status', 'Receipt updated successfully.');
        $this->closeModal();
    }

    // Receipt Merging Methods
    public function toggleReceiptSelection(int $receiptId): void
    {
        if (in_array($receiptId, $this->selectedReceipts)) {
            $this->selectedReceipts = array_filter($this->selectedReceipts, fn($id) => $id !== $receiptId);
        } else {
            $this->selectedReceipts[] = $receiptId;
        }
        
        $this->showMatchButton = count($this->selectedReceipts) >= 1;
    }

    public function matchSelectedReceipts(): void
    {
        if (count($this->selectedReceipts) < 1) {
            session()->flash('error', 'Please select at least 1 receipt to match.');
            return;
        }

        if ($this->isMatching) {
            session()->flash('error', 'A matching operation is already in progress. Please wait.');
            return;
        }

        try {
            $this->isMatching = true;
            $selectedCount = count($this->selectedReceipts);
            
            // Dispatch matching jobs for each selected receipt
            foreach ($this->selectedReceipts as $receiptId) {
                MatchReceipt::dispatch($receiptId);
            }
            
            // Clear selections immediately
            $this->selectedReceipts = [];
            $this->showMatchButton = false;
            
            session()->flash('status', "Matching {$selectedCount} receipts in background. They will be matched to statement lines shortly.");
            
        } catch (\Exception $e) {
            session()->flash('error', 'Failed to start matching process: ' . $e->getMessage());
        } finally {
            $this->isMatching = false;
        }
    }

    public function clearSelection(): void
    {
        $this->selectedReceipts = [];
        $this->showMatchButton = false;
    }

    public function isReceiptSelected(int $receiptId): bool
    {
        return in_array($receiptId, $this->selectedReceipts);
    }


    // Manual Matching Methods
    public function openManualMatchModal(int $receiptId): void
    {
        $this->manualMatchReceiptId = $receiptId;
        $this->manualMatchTransactionId = null;
        $this->showManualMatchModal = true;
        $this->manualMatchSearch = '';
        
        // Force Livewire to re-render
        $this->render();
        
        session()->flash('status', "Modal opened for receipt {$receiptId}");
    }

    public function closeManualMatchModal(): void
    {
        $this->showManualMatchModal = false;
        $this->manualMatchReceiptId = null;
        $this->manualMatchTransactionId = null;
        $this->manualMatchSearch = '';
    }

    public function setManualMatchTransaction(int $transactionId): void
    {
        $this->manualMatchTransactionId = $transactionId;
    }

    public function setManualMatchSort(string $sortBy): void
    {
        if ($this->manualMatchSortBy === $sortBy) {
            $this->manualMatchSortOrder = $this->manualMatchSortOrder === 'asc' ? 'desc' : 'asc';
        } else {
            $this->manualMatchSortBy = $sortBy;
            $this->manualMatchSortOrder = 'desc';
        }
    }

    public function confirmManualMatch(): void
    {
        if (!$this->manualMatchReceiptId || !$this->manualMatchTransactionId) {
            session()->flash('error', 'Please select both a receipt and a transaction to match.');
            return;
        }

        try {
            $receipt = Receipt::find($this->manualMatchReceiptId);
            $transaction = StatementTransaction::find($this->manualMatchTransactionId);

            if (!$receipt || !$transaction) {
                session()->flash('error', 'Receipt or transaction not found.');
                return;
            }

            // Check if receipt is already matched
            if ($receipt->matches()->exists()) {
                session()->flash('error', 'This receipt is already matched to a transaction.');
                return;
            }

            // Create the manual match
            \App\Models\TransactionMatch::create([
                'statement_transaction_id' => $transaction->id,
                'receipt_id' => $receipt->id,
                'matched_amount' => $receipt->total_amount,
                'match_method' => 'manual',
                'amount_confidence' => 1.0, // Perfect confidence for manual matches
                'date_confidence' => 1.0,
                'merchant_confidence' => 1.0,
                'explanation' => 'Manually matched by user',
            ]);

            // Trigger re-classification with receipt information
            \App\Jobs\ReclassifyWithReceipt::dispatch($transaction, $receipt)->delay(now()->addSeconds(10));

            session()->flash('status', "Successfully matched receipt '{$receipt->merchant_name}' to transaction '{$transaction->merchant_name}'.");

            $this->closeManualMatchModal();

        } catch (\Exception $e) {
            session()->flash('error', 'Failed to create manual match: ' . $e->getMessage());
        }
    }

    public function getAvailableTransactions()
    {
        if (!$this->manualMatchReceiptId) {
            return collect();
        }

        $receipt = Receipt::find($this->manualMatchReceiptId);
        if (!$receipt) {
            return collect();
        }

        // Get all open statements
        $statements = Statement::where('status', 'open')->get();
        if ($statements->isEmpty()) {
            return collect();
        }

        // Build query
        $query = StatementTransaction::whereIn('statement_id', $statements->pluck('id'));

        // Apply search filter
        if (!empty($this->manualMatchSearch)) {
            $searchTerm = '%' . $this->manualMatchSearch . '%';
            $query->where(function($q) use ($searchTerm) {
                $q->where('merchant_name', 'like', $searchTerm)
                  ->orWhere('description', 'like', $searchTerm)
                  ->orWhere('amount', 'like', $searchTerm);
            });
        }

        // Apply sorting
        switch ($this->manualMatchSortBy) {
            case 'date':
                $query->orderBy('transaction_date', $this->manualMatchSortOrder);
                break;
            case 'amount':
                $query->orderBy('amount', $this->manualMatchSortOrder);
                break;
            case 'merchant':
                $query->orderBy('merchant_name', $this->manualMatchSortOrder);
                break;
        }

        return $query->get();
    }
}


