<?php

namespace App\Livewire\Admin;

use App\Models\Order;
use Livewire\Component;

class OrderDetail extends Component
{
    public $orderId;
    public $order;
    public $editing = false;
    
    // Editable fields
    public $status;
    public $trackingNumber;
    public $notes;
    public $zohoInvoiceNumber;
    public $supplierInvoiceNumber;
    public $invoicedSubtotal;
    public $invoicedTaxAmount;
    public $invoicedShippingAmount;
    public $invoicedTotalAmount;

    public function mount($orderId)
    {
        $this->orderId = $orderId;
        $this->loadOrder();
    }

    public function loadOrder()
    {
        $this->order = Order::with(['user', 'orderLines.colourway'])
            ->findOrFail($this->orderId);
        
        $this->loadFormData();
    }

    public function loadFormData()
    {
        $this->status = $this->order->status;
        $this->trackingNumber = $this->order->tracking_number;
        $this->notes = $this->order->notes;
        $this->zohoInvoiceNumber = $this->order->zoho_invoice_number;
        $this->supplierInvoiceNumber = $this->order->supplier_invoice_number;
        $this->invoicedSubtotal = $this->order->invoiced_subtotal;
        $this->invoicedTaxAmount = $this->order->invoiced_tax_amount;
        $this->invoicedShippingAmount = $this->order->invoiced_shipping_amount;
        $this->invoicedTotalAmount = $this->order->invoiced_total_amount;
    }

    public function toggleEdit()
    {
        if ($this->editing) {
            $this->editing = false;
            $this->loadFormData(); // Reset form data if canceling
        } else {
            $this->editing = true;
        }
    }

    public function save()
    {
        $this->validate([
            'status' => 'required|in:pending,confirmed,processing,shipped,delivered,cancelled',
            'trackingNumber' => 'nullable|string|max:255',
            'notes' => 'nullable|string',
            'zohoInvoiceNumber' => 'nullable|string|max:255',
            'supplierInvoiceNumber' => 'nullable|string|max:255',
            'invoicedSubtotal' => 'nullable|numeric|min:0',
            'invoicedTaxAmount' => 'nullable|numeric|min:0',
            'invoicedShippingAmount' => 'nullable|numeric|min:0',
            'invoicedTotalAmount' => 'nullable|numeric|min:0',
        ]);

        $updateData = [
            'status' => $this->status,
            'tracking_number' => $this->trackingNumber,
            'notes' => $this->notes,
            'zoho_invoice_number' => $this->zohoInvoiceNumber,
            'supplier_invoice_number' => $this->supplierInvoiceNumber,
            'invoiced_subtotal' => $this->invoicedSubtotal,
            'invoiced_tax_amount' => $this->invoicedTaxAmount,
            'invoiced_shipping_amount' => $this->invoicedShippingAmount,
            'invoiced_total_amount' => $this->invoicedTotalAmount,
        ];

        // Set shipped_at timestamp if status changed to shipped
        if ($this->status === 'shipped' && $this->order->status !== 'shipped') {
            $updateData['shipped_at'] = now();
        }

        // Set delivered_at timestamp if status changed to delivered
        if ($this->status === 'delivered' && $this->order->status !== 'delivered') {
            $updateData['delivered_at'] = now();
        }

        $this->order->update($updateData);
        
        $this->editing = false;
        $this->loadOrder();
        
        session()->flash('message', 'Order updated successfully!');
    }

    public function deleteOrder()
    {
        $orderNumber = $this->order->order_number;
        
        // Delete order lines
        $this->order->orderLines()->delete();
        
        // Delete the order
        $this->order->delete();
        
        session()->flash('message', "Order #{$orderNumber} has been deleted successfully!");
        
        return redirect()->route('admin.orders');
    }

    public function render()
    {
        return view('livewire.admin.order-detail')
            ->layout('layouts.admin')
            ->title('Order #' . $this->order->order_number);
    }
}

