<?php

namespace Tests\Feature;

use App\Models\LibraryTrack;
use App\Models\CustomSession;
use App\Services\MediaService;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Support\Facades\Storage;
use Tests\TestCase;

class MediaManagementTest extends TestCase
{
    use RefreshDatabase;

    protected function setUp(): void
    {
        parent::setUp();
        Storage::fake('local');
    }

    /** @test */
    public function it_can_generate_signed_urls_for_library_tracks()
    {
        // Create a track with a file path
        $track = LibraryTrack::factory()->create([
            'file_path' => 'tracks/test-track.mp3'
        ]);

        // Mock the media service to return a signed URL
        $this->mock(MediaService::class, function ($mock) {
            $mock->shouldReceive('generateSignedUrl')
                ->once()
                ->andReturn('http://localhost/media/signed/test-token');
        });

        // Access the audio_url attribute
        $audioUrl = $track->audio_url;

        $this->assertNotNull($audioUrl);
        $this->assertStringContainsString('/media/signed/', $audioUrl);
    }

    /** @test */
    public function it_can_generate_signed_urls_for_custom_sessions()
    {
        // Create a custom session with a file path
        $session = CustomSession::factory()->create([
            'file_path' => 'custom-sessions/test-session.mp3',
            'status' => 'ready'
        ]);

        // Mock the media service to return a signed URL
        $this->mock(MediaService::class, function ($mock) {
            $mock->shouldReceive('generateSignedUrl')
                ->once()
                ->andReturn('http://localhost/media/signed/test-token');
        });

        // Access the audio_url attribute
        $audioUrl = $session->audio_url;

        $this->assertNotNull($audioUrl);
        $this->assertStringContainsString('/media/signed/', $audioUrl);
    }

    /** @test */
    public function it_returns_null_for_audio_url_when_no_file_path()
    {
        $track = LibraryTrack::factory()->create([
            'file_path' => null
        ]);

        $this->assertNull($track->audio_url);
    }

    /** @test */
    public function it_returns_null_for_audio_url_when_file_path_is_empty()
    {
        $track = LibraryTrack::factory()->create([
            'file_path' => ''
        ]);

        $this->assertNull($track->audio_url);
    }

    /** @test */
    public function it_can_validate_signed_url_tokens()
    {
        $mediaService = app(MediaService::class);
        
        // Create a test token
        $token = 'test-token-123';
        $filePath = 'tracks/test.mp3';
        
        // Store the token in cache
        cache()->put("signed_url_{$token}", [
            'file_path' => $filePath,
            'expires' => now()->addMinutes(60)->timestamp
        ], 3600);

        // Validate the token
        $result = $mediaService->validateSignedUrlToken($token);

        $this->assertNotNull($result);
        $this->assertEquals($filePath, $result['file_path']);
    }

    /** @test */
    public function it_rejects_expired_signed_url_tokens()
    {
        $mediaService = app(MediaService::class);
        
        // Create an expired token
        $token = 'expired-token-123';
        $filePath = 'tracks/test.mp3';
        
        // Store the token in cache with expired timestamp
        cache()->put("signed_url_{$token}", [
            'file_path' => $filePath,
            'expires' => now()->subMinutes(60)->timestamp
        ], 3600);

        // Validate the token
        $result = $mediaService->validateSignedUrlToken($token);

        $this->assertNull($result);
    }

    /** @test */
    public function it_rejects_invalid_signed_url_tokens()
    {
        $mediaService = app(MediaService::class);
        
        // Try to validate a non-existent token
        $result = $mediaService->validateSignedUrlToken('non-existent-token');

        $this->assertNull($result);
    }
}
