<?php

namespace App\Http\Livewire\Finance\CommissionCashflow;

use App\Models\Seasons;
use Livewire\Component;
use App\Models\Cashflow;
use App\Models\Customer;
use App\Models\Countries;
use App\Models\Suppliers;
use App\Helper\Conversions;
use App\Models\Departments;
use App\Models\CashflowLine;
use App\Models\ShipmentLine;
use Livewire\Attributes\Computed;
use Illuminate\Support\Facades\DB;

class CoCashflowNew extends Component
{
    public $name;
    public $exfty_date_to, $customer_id, $factory_id, $country_id, $department_id, $category_id;

    protected $rules = [
        'exfty_date_to' => 'required|date',
        'customer_id' => 'nullable|exists:customers,id',
        'factory_id' => 'nullable|exists:suppliers,id',
        'country_id' => 'nullable|exists:countries,id',
        'department_id' => 'nullable|exists:departments,id',
        'category_id' => 'nullable|in:ladies,mens,childrens,accessories',
    ];

    #[Computed]
    public function customers()
    {
        return Customer::allCached();
    }

    #[Computed]
    public function factories()
    {
        return Suppliers::allCached();
    }

    #[Computed]
    public function countries()
    {
        return Countries::all();
    }

    #[Computed]
    public function departments()
    {
        return Departments::allCached();
    }
    #[Computed]
    public function seasons()
    {
        return Seasons::allCached();
    }
    // public function render()
    // {
    //     return view('livewire.finance.cashflow.wh-cashflow-new');
    // }

    public function refreshData(){
        session()->flash('message', 'Validation Error');
		session()->flash('alert-class', 'alert-warning');
        $this->validate();
        $results = $this->getResults();
        $this->extractResults($results);
        $this->dispatch('close-modal');
        $this->dispatch('refresh-page');
        session()->flash('message', 'Created');
		session()->flash('alert-class', 'alert-success');
    }

    public function extractResults($results){
        $cashflow = Cashflow::create([
            'name' => $this->name,
            'data' => json_encode([
                'exfty_date_to' => $this->exfty_date_to,
                'customer' => Customer::find($this->customer_id)?->name,
                'factory' => Suppliers::find($this->factory_id)?->name,
                'country' => Countries::find($this->country_id)?->country,
                'department' => Departments::find($this->department_id)?->description,
                'category' => $this->category_id,
            ]),
        ]);
        $lines = [];
        foreach($results as $result){
            $line = new CashflowLine;
            $line->cashflows_id = $cashflow->id;
            $line->shipment_lines_id = $result->id;
            $line->cmt = $result->total_cmt;
            $line->cmt_terms = $result->fact_terms;
            $line->cmt_currency = $result->fact_currency;
            $line->yarn = $result->total_yarn;
            $line->yarn_terms = $result->yarn_terms;
            $line->yarn_currency = '£';
            $line->sale = $result->total_quote;
            $line->sale_terms = $result->customer_terms;
            $line->sale_currency = $result->cust_currency;
            $line->vat = $this->checkVAT($result);
            $line->factored = $result->customer_factored ?? FALSE;
            $line->seasons_id = $result->seasons_id;
            $line->suppliers_id = $result->factory_id;
            $line->customers_id = $result->customer_id;
            $line->designs_id = $result->designs_id;
            $line->customer_po = $result->customer_po;
            $line->customer_ref = $result->customer_ref;
            $line->description = $result->description;
            $line->colour = $result->colour;
            $line->incoterms = $result->incoterms;
            $line->exfty = $result->exfty;
            $line->qty = $result->total_qty;
            $line->is_full_factored = $result->is_full_factored;
            $line->is_complete = $result->complete;
            $lines[] = $line;
        }
        $cashflow->cashflow_lines()->saveMany($lines);
    }

    public function checkVAT($result){
        // dd($result);
        if($result->incoterms == 'CIF'
            || $result->incoterms == 'DAP'
            || $result->incoterms == 'DDP'
            || $result->incoterms == 'DPU'
            || $result->incoterms == 'CFR'
            || $result->incoterms == 'CIP'
            || $result->incoterms == 'CPT')
        {
            return $result->total_quote * 0.2;
        }
        else
        {
            return 0;
        }
    }

    public function getResults(){
        $query = ShipmentLine::select(
            "shipment_lines.id AS id",
            "seasons.id AS seasons_id",
            "factory.id AS factory_id",
            "customers.id AS customer_id",
            "styles.designs_id",
            "customer_orders.customer_po AS customer_po",
            "styles.customer_ref",
            "designs.description AS description",
            "colourways.name AS colour",
            "customer_orders.incoterms AS incoterms",

            "shipment_lines.complete AS complete",

            "factory.payment_terms AS fact_terms",
            "factory.currency AS fact_currency",

            "customers.payment_terms AS customer_terms",
            "customers.currency AS cust_currency",

            "customers.factored AS customer_factored",



        )
        ->selectRaw("
            CASE
                WHEN factory_cust_date > exfty
                THEN factory_cust_date
                ELSE exfty
            END AS exfty")
        ->selectRaw("SUM(sls.qty) AS total_qty")
        ->selectRaw("SUM(sls.shipped_qty) AS total_shipped")
        ->selectRaw("
            (select
                (
                    case when (
                    min(`sl`.`collection_date`) is not null
                    ) then min(`sl`.`collection_date`) else min(`sl`.`exfty`) end
                )
                from
                `shipment_lines` `sl`
                where
                (
                    `sl`.`shipment_id` = `shipment_lines`.`shipment_id`
                )
            ) AS `truck_first_collection`
        ")
        ->selectRaw("
            (
                select
                concat(
                    '[',
                    group_concat(
                    JSON_OBJECT(
                        'size', sizes.name,
                        'sizes_id', sizes.id,
                        'qty', sls.qty,
                        'shipped_qty', sls.shipped_qty,
                        'price', COALESCE((

                        (

                            WITH ranked_prices AS (
                                SELECT
                                p.*,
                                ROW_NUMBER() OVER (PARTITION BY style_versions_id, sizes_id, colourways_id, colour_type ORDER BY quote_status DESC) AS row_num
                                FROM prices AS p
                                WHERE style_versions_id = style_versions.id
                                AND (p.sizes_id = sizes.id OR p.sizes_id = 1 OR sizes_id IS NULL)
                                AND (p.colourways_id = colourways.id OR p.colourways_id IS NULL)
                                AND (p.colour_type = colourways.colour_type OR p.colour_type IS NULL)
                                AND deleted_at IS NULL
                            )
                            SELECT
                                CASE
                                WHEN (quote_status = 'confirmed' AND row_num = 1) OR (quote_status != 'confirmed' AND COUNT(p.id) = 1) THEN
                                    json_object(
                                    'id', p.id,
                                    'gmt_trans_currency', p.gmt_trans_currency,
                                    'transport_budget', p.gmt_trans * COALESCE(NULLIF(sls.shipped_qty, ''), sls.qty),
                                    'total_quote', p.quote * COALESCE(NULLIF(sls.shipped_qty, ''), sls.qty),
                                    'total_cmt', p.cmt * COALESCE(NULLIF(sls.shipped_qty, ''), sls.qty),
                                    'cmt', p.cmt,
                                    'quote', p.quote,
                                    'quote_status', p.quote_status,
                                    'cmt_status', p.cmt_status,
                                    'sizes_id', p.sizes_id,
                                    'colourways_id', p.colourways_id,
                                    'colour_type', p.colour_type,
                                    'model', p.model,
                                    'total_yarn',
                                    (
                                        CASE WHEN (model = 'standard') THEN (
                                          -- yarn_cost_kg_eur will be computed separately as it requires loading relationships
                                          0
                                        ) + yarn_trans -- currency conversion will be handled in PHP
                                        ) WHEN (model = 'manual') THEN p.yarn -- currency conversion will be handled in PHP
                                        ) + yarn_trans -- currency conversion will be handled in PHP
                                        ) END
                                      ) * costed_weight  * COALESCE(NULLIF(sls.shipped_qty, ''), sls.qty)

                                    )
                                ELSE 'MANY PRICES'
                                END AS result
                            FROM ranked_prices AS p
                            GROUP BY p.id
                            LIMIT 1

                        )

                        ),'NO PRICES'),
                        'discount', colq.discount,
                        'commission', colq.commission
                    ) SEPARATOR ','
                    ),
                    ']'
                )
                from  shipment_line_sizes sls
                join sizes on sizes.id = sls.sizes_id and sls.shipment_line_id = shipment_lines.id
                join customer_order_line_quantities colq on colq.customer_order_lines_id = customer_order_lines.id and colq.sizes_id = sls.sizes_id
                group by sls.shipment_line_id
            ) AS sizes
        ")
        ->selectRaw("(select payment_terms from colourway_yarns cy join yarn_colours yc on yc.id = cy.yarn_colours_id join yarns y on y.id = yc.yarn_id join suppliers s on s.id = y.suppliers_id limit 1) as yarn_terms")
        ->join('customer_order_lines', 'customer_order_lines.id', '=', 'shipment_lines.customer_order_lines_id')
        ->Join('customer_orders', function($join)
        {
            $join->on('customer_orders.id', '=', 'customer_order_lines.customer_orders_id');
            $join->on('customer_orders.order_type', '=', DB::raw("'commission'"));
            $join->on('customer_orders.cancelled', '=', DB::raw(0));
        })
        ->join('customers', 'customers.id', '=', 'customer_orders.customers_id')
        ->join('colourways', 'colourways.id', '=', 'customer_order_lines.colourways_id')
        ->join('style_versions', 'style_versions.id', '=', 'colourways.style_versions_id')
        ->join('styles', 'styles.id', '=', 'style_versions.styles_id')
        ->join('seasons', 'seasons.id', '=', 'customer_orders.seasons_id')
        ->join('suppliers as factory', 'factory.id', '=', 'style_versions.factory_id')
        ->join('designs', 'designs.id', '=', 'styles.designs_id')
        ->leftJoin('shipments', 'shipments.id', '=', 'shipment_lines.shipment_id')
        ->leftJoin('suppliers as transporters', 'transporters.id', '=', 'shipments.transporter_id')
        ->leftJoin('customer_order_line_quantities as colq', 'colq.customer_order_lines_id', '=', 'customer_order_lines.id')
        ->leftJoin('shipment_line_sizes as sls', function($join)
        {
            $join->on('sls.shipment_line_id', '=', 'shipment_lines.id');
            $join->on('sls.sizes_id', '=', 'colq.sizes_id');
        })
        ->groupBy('shipment_lines.id')
        // ->where('shipment_lines.id', 7205)
        ->whereNull('shipment_lines.deleted_at');


        $query->whereNull('shipment_lines.rt_invoice');

        if(!empty($this->exfty_date_to)){
			$query->where('shipment_lines.exfty', '<=', $this->exfty_date_to);
		}
		if(!empty($this->customer_id)){
			$query->where('customer_orders.customers_id', $this->customer_id);
		}
		if(!empty($this->factory_id)){
			$query->where('style_versions.factory_id', $this->factory_id);
		}
		if(!empty($this->country_id)){
			$query->where('factory.countries_id', $this->country_id);
		}
		if(!empty($this->department_id)){
			$query->where('styles.departments_id', $this->department_id);
		}
		if(!empty($this->category)){
			$query->where('styles.category', $this->category);
		}
        if(!empty($this->truckAssigned)){
			$query->whereNull('shipment_lines.shipment_id');
		}


        $query->orderByRaw("
            -(shipment_lines.complete),
            -(truck_first_collection) desc,
            shipment_lines.shipment_id,
            -(shipment_lines.exfty) desc,
            factory.countries_id,
            -(factory.name),
            -(customers.name),
            -(
            customer_orders.customer_po
            ),
            -(styles.id),
            -(colourways.name)
        ");

        $results = $query->get();

        // dd(json_decode($results[70]));

		foreach($results as $l=>$line){
            $model = "";
			$results[$l]->sizes = collect(json_decode($line->sizes));
            foreach($line->sizes as $p=>$price){
                $results[$l]->sizes[$p]->price = json_decode($price->price);
            }
            foreach($line->sizes as $p=>$price){
                if ($model == "") {
                    $model = $price->price?->model;
                }
                elseif($model != $price->price?->model ?? ""){
                    $model = "MIXED";
                    // dd($price->price?->model, $model);
                }
            }
            $results[$l]->total_quote = $line->sizes->sum('price.total_quote');
            $results[$l]->total_cmt = $line->sizes->sum('price.total_cmt');
            $results[$l]->total_yarn = $line->sizes->sum('price.total_yarn');
            $results[$l]->is_full_factored = $model == 'full_factored' ? TRUE : FALSE;
            // if($line->id == 5569)
            //     dd($model, $results[$l]->is_full_factored, $results[$l]->sizes);
		}

        // dd($results->first());

        return $results;
    }
}
