<?php

namespace App\Http\Controllers\Api\V1;

use App\Http\Controllers\Controller;
use App\Models\Profile;
use App\Models\Consent;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Http\Response;
use Illuminate\Support\Facades\Auth;
use Illuminate\Validation\Rule;

class ProfileController extends Controller
{
    public function show(): JsonResponse
    {
        $user = Auth::user();
        $profile = $user->profile;
        $consents = $user->consents;
        
        return response()->json([
            'data' => [
                'user' => [
                    'id' => $user->id,
                    'name' => $user->name,
                    'email' => $user->email,
                    'emailVerifiedAt' => $user->email_verified_at,
                ],
                'profile' => $profile ? [
                    'wellnessGoals' => $profile->wellness_goals,
                    'voicePreference' => $profile->voice_preference,
                    'sensitivityLevel' => $profile->sensitivity_level,
                    'medicalNotes' => $profile->medical_notes,
                    'preferences' => $profile->preferences,
                ] : null,
                'consents' => $consents->map(fn($consent) => [
                    'type' => $consent->type,
                    'description' => $consent->description,
                    'version' => $consent->version,
                    'isAccepted' => $consent->is_accepted,
                    'acceptedAt' => $consent->accepted_at,
                    'expiresAt' => $consent->expires_at,
                    'isActive' => $consent->isActive(),
                ]),
            ],
        ]);
    }

    public function update(Request $request): JsonResponse
    {
        $user = Auth::user();
        
        $request->validate([
            'name' => 'sometimes|string|max:255',
            'wellnessGoals' => 'sometimes|array',
            'wellnessGoals.*' => 'string|max:100',
            'voicePreference' => ['sometimes', Rule::in(['female', 'male', 'neutral'])],
            'sensitivityLevel' => ['sometimes', Rule::in(['low', 'standard', 'high'])],
            'medicalNotes' => 'sometimes|nullable|string|max:1000',
            'preferences' => 'sometimes|array',
        ]);
        
        if ($request->has('name')) {
            $user->update(['name' => $request->name]);
        }
        
        $profile = $user->profile;
        if (!$profile) {
            $profile = Profile::create(['user_id' => $user->id]);
        }
        
        $profile->update([
            'wellness_goals' => $request->wellnessGoals ?? $profile->wellness_goals,
            'voice_preference' => $request->voicePreference ?? $profile->voice_preference,
            'sensitivity_level' => $request->sensitivityLevel ?? $profile->sensitivity_level,
            'medical_notes' => $request->medicalNotes ?? $profile->medical_notes,
            'preferences' => $request->preferences ?? $profile->preferences,
        ]);
        
        return response()->json([
            'message' => 'Profile updated successfully',
            'data' => [
                'wellnessGoals' => $profile->wellness_goals,
                'voicePreference' => $profile->voice_preference,
                'sensitivityLevel' => $profile->sensitivity_level,
                'medicalNotes' => $profile->medical_notes,
                'preferences' => $profile->preferences,
            ],
        ]);
    }

    public function updateConsent(Request $request): JsonResponse
    {
        $request->validate([
            'type' => 'required|string',
            'isAccepted' => 'required|boolean',
        ]);
        
        $user = Auth::user();
        $consent = $user->consents()->where('type', $request->type)->first();
        
        if (!$consent) {
            return response()->json([
                'error' => 'Consent type not found',
                'message' => 'The specified consent type does not exist.',
            ], Response::HTTP_NOT_FOUND);
        }
        
        if ($request->isAccepted) {
            $consent->accept();
        } else {
            $consent->revoke();
        }
        
        return response()->json([
            'message' => 'Consent updated successfully',
            'data' => [
                'type' => $consent->type,
                'isAccepted' => $consent->is_accepted,
                'acceptedAt' => $consent->accepted_at,
                'isActive' => $consent->isActive(),
            ],
        ]);
    }
}
