<?php

namespace App\Http\Controllers\Api\V1;

use App\Http\Controllers\Controller;
use App\Models\Consent;
use App\Models\Profile;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Auth;

class ProfileController extends Controller
{
    public function show(): JsonResponse
    {
        $user = Auth::user();
        $profile = $user->profile;
        $consents = $user->consents;

        return response()->json([
            'user' => [
                'id' => $user->id,
                'name' => $user->name,
                'email' => $user->email,
                'emailVerifiedAt' => $user->email_verified_at,
            ],
            'profile' => $profile ? [
                'goals' => $profile->goals,
                'voicePreference' => $profile->voice_preference,
                'sensitivityLevel' => $profile->sensitivity_level,
                'wellnessInterests' => $profile->wellness_interests,
                'timezone' => $profile->timezone,
                'notificationsEnabled' => $profile->notifications_enabled,
            ] : null,
            'consents' => $consents ? [
                'type' => $consents->type,
                'description' => $consents->description,
                'granted' => $consents->granted,
                'grantedAt' => $consents->granted_at,
                'revokedAt' => $consents->revoked_at,
            ] : null,
        ]);
    }

    public function update(Request $request): JsonResponse
    {
        $request->validate([
            'goals' => 'nullable|string|max:1000',
            'voicePreference' => 'nullable|string|in:male,female,neutral',
            'sensitivityLevel' => 'nullable|string|in:low,medium,high',
            'wellnessInterests' => 'nullable|array',
            'wellnessInterests.*' => 'string|max:100',
            'timezone' => 'nullable|string|max:50',
            'notificationsEnabled' => 'nullable|boolean',
        ]);

        $user = Auth::user();
        $profile = $user->profile;

        if (!$profile) {
            $profile = Profile::create([
                'user_id' => $user->id,
                'goals' => $request->goals,
                'voice_preference' => $request->voicePreference,
                'sensitivity_level' => $request->sensitivityLevel,
                'wellness_interests' => $request->wellnessInterests,
                'timezone' => $request->timezone,
                'notifications_enabled' => $request->notificationsEnabled ?? true,
            ]);
        } else {
            $profile->update([
                'goals' => $request->goals,
                'voice_preference' => $request->voicePreference,
                'sensitivity_level' => $request->sensitivityLevel,
                'wellness_interests' => $request->wellnessInterests,
                'timezone' => $request->timezone,
                'notifications_enabled' => $request->notificationsEnabled,
            ]);
        }

        return response()->json([
            'message' => 'Profile updated successfully',
            'profile' => [
                'goals' => $profile->goals,
                'voicePreference' => $profile->voice_preference,
                'sensitivityLevel' => $profile->sensitivity_level,
                'wellnessInterests' => $profile->wellness_interests,
                'timezone' => $profile->timezone,
                'notificationsEnabled' => $profile->notifications_enabled,
            ],
        ]);
    }

    public function consents(): JsonResponse
    {
        $user = Auth::user();
        $consents = $user->consents;

        return response()->json([
            'consents' => $consents ? [
                'type' => $consents->type,
                'description' => $consents->description,
                'granted' => $consents->granted,
                'grantedAt' => $consents->granted_at,
                'revokedAt' => $consents->revoked_at,
            ] : null,
        ]);
    }

    public function updateConsents(Request $request): JsonResponse
    {
        $request->validate([
            'type' => 'required|string|max:100',
            'description' => 'required|string|max:500',
            'granted' => 'required|boolean',
        ]);

        $user = Auth::user();
        $consents = $user->consents;

        if (!$consents) {
            $consents = Consent::create([
                'user_id' => $user->id,
                'type' => $request->type,
                'description' => $request->description,
                'granted' => $request->granted,
                'granted_at' => $request->granted ? now() : null,
            ]);
        } else {
            $consents->update([
                'type' => $request->type,
                'description' => $request->description,
                'granted' => $request->granted,
                'granted_at' => $request->granted ? now() : null,
                'revoked_at' => $request->granted ? null : now(),
            ]);
        }

        return response()->json([
            'message' => 'Consents updated successfully',
            'consents' => [
                'type' => $consents->type,
                'description' => $consents->description,
                'granted' => $consents->granted,
                'grantedAt' => $consents->granted_at,
                'revokedAt' => $consents->revoked_at,
            ],
        ]);
    }
}
