<?php

namespace App\Services\Finance;

use App\Models\Receipt;
use App\Models\ReceiptLine;
use App\Models\VatAnalysis;
use App\Services\VAT\UkVatService;
use Illuminate\Support\Facades\Log;

class VatProcessingService
{
    public function __construct(private UkVatService $vatService) {}

    /**
     * Process VAT analysis for a receipt when it's submitted to finance
     * This uses the comprehensive OCR data that was extracted earlier
     */
    public function processReceiptVat(Receipt $receipt): void
    {
        $startTime = microtime(true);
        
        Log::info("Starting VAT processing for receipt", [
            'receipt_id' => $receipt->id,
            'lines_count' => $receipt->lines->count()
        ]);

        foreach ($receipt->lines as $line) {
            $this->processLineVat($line);
        }
        
        $endTime = microtime(true);
        $processingTime = round($endTime - $startTime, 2);
        
        Log::info("VAT processing completed", [
            'receipt_id' => $receipt->id,
            'processing_time' => $processingTime . 's'
        ]);
    }

    /**
     * Process VAT analysis for a single line using OCR data
     */
    private function processLineVat(ReceiptLine $line): void
    {
        try {
            // Use the comprehensive VAT data from OCR if available
            $ocrVatData = $this->extractVatDataFromOcr($line);
            
            if ($ocrVatData) {
                // Use OCR-extracted VAT data (most accurate)
                $this->createVatAnalysisFromOcr($line, $ocrVatData);
            } else {
                // Fallback to LLM VAT analysis
                $this->vatService->analyzeLine($line);
            }
        } catch (\Exception $e) {
            Log::error('VAT processing failed for line', [
                'line_id' => $line->id,
                'error' => $e->getMessage()
            ]);
        }
    }

    /**
     * Extract VAT data from OCR meta data
     */
    private function extractVatDataFromOcr(ReceiptLine $line): ?array
    {
        $meta = $line->meta ?? [];
        
        // Check if OCR extracted VAT information
        if (isset($meta['vat_rate']) || isset($meta['vat_amount']) || isset($meta['vat_status'])) {
            return [
                'vat_rate' => $meta['vat_rate'] ?? 0,
                'vat_amount' => $meta['vat_amount'] ?? 0,
                'vat_status' => $meta['vat_status'] ?? null,
                'vat_code' => $meta['vat_code'] ?? null,
                'is_vat_inclusive' => $meta['is_vat_inclusive'] ?? true,
                'net_amount' => $meta['net_amount'] ?? null,
            ];
        }
        
        return null;
    }

    /**
     * Create VAT analysis using OCR data (most accurate)
     */
    private function createVatAnalysisFromOcr(ReceiptLine $line, array $vatData): void
    {
        $vatRate = (float) ($vatData['vat_rate'] ?? 0);
        $vatAmount = (float) ($vatData['vat_amount'] ?? 0);
        $isVatInclusive = $vatData['is_vat_inclusive'] ?? true;
        
        // Calculate claimable VAT amount
        $claimable = $line->is_personal ? 0.0 : $vatAmount;
        
        // If no VAT amount from OCR, calculate it
        if (!$vatAmount && $vatRate > 0) {
            if ($isVatInclusive) {
                $claimable = round(($line->line_total * ($vatRate / (100 + $vatRate))), 2);
            } else {
                $claimable = round(($line->line_total * ($vatRate / 100)), 2);
            }
        }
        
        VatAnalysis::updateOrCreate(
            ['receipt_line_id' => $line->id],
            [
                'vat_rate' => $vatRate,
                'vat_amount_claimable' => $claimable,
                'reasoning' => 'OCR-extracted VAT data (most accurate)',
            ]
        );
    }

    /**
     * Process VAT for multiple receipts in batch
     */
    public function processBatchVat(array $receiptIds): void
    {
        $receipts = Receipt::whereIn('id', $receiptIds)->with('lines')->get();
        
        foreach ($receipts as $receipt) {
            $this->processReceiptVat($receipt);
        }
    }
}
