<?php

namespace App\Services;

use App\Contracts\MemoryService as MemoryServiceContract;
use App\Models\AiContextSummary;
use App\Models\User;
use App\Models\UserCheckin;
use Illuminate\Support\Str;

class MemoryServiceImpl implements MemoryServiceContract
{
    public function createMonthlySummary(User $user, ?\DateTimeInterface $monthStart = null): AiContextSummary
    {
        $monthStart = $monthStart ? \Carbon\CarbonImmutable::instance(\Carbon\Carbon::parse($monthStart)) : now()->startOfMonth();
        $monthEnd = $monthStart->endOfMonth();

        $checkins = UserCheckin::query()
            ->where('user_id', $user->id)
            ->whereBetween('created_at', [$monthStart, $monthEnd])
            ->orderBy('created_at')
            ->pluck('note')
            ->all();

        $summaryText = $this->summarizeNotes($checkins);

        return AiContextSummary::create([
            'id' => (string) Str::uuid(),
            'user_id' => $user->id,
            'summary' => $summaryText,
            'source_window' => [
                'from' => $monthStart->toDateString(),
                'to' => $monthEnd->toDateString(),
            ],
            'created_at' => now(),
        ]);
    }

    public function rotateSummaries(User $user, int $keep = 12): int
    {
        $idsToDelete = AiContextSummary::query()
            ->where('user_id', $user->id)
            ->orderByDesc('created_at')
            ->skip($keep)
            ->take(PHP_INT_MAX)
            ->pluck('id');
        return AiContextSummary::whereIn('id', $idsToDelete)->delete();
    }

    private function summarizeNotes(array $notes): string
    {
        if (empty($notes)) {
            return 'No check-ins this period.';
        }

        $joined = implode("\n- ", array_map(fn ($n) => Str::limit(trim((string) $n), 240), $notes));
        return "Highlights:\n- " . $joined;
    }
}


