# PostgreSQL Setup Guide

This application now uses PostgreSQL as its primary database. Follow these instructions to set up your database.

## Local Development Setup

### 1. Install PostgreSQL

**macOS (using Homebrew):**
```bash
brew install postgresql@16
brew services start postgresql@16
```

**Ubuntu/Debian:**
```bash
sudo apt update
sudo apt install postgresql postgresql-contrib
sudo systemctl start postgresql
```

**Windows:**
Download and install from [postgresql.org](https://www.postgresql.org/download/windows/)

### 2. Create Database and User

```bash
# Connect to PostgreSQL as the default postgres user
psql postgres

# Create the database
CREATE DATABASE photowebsite;

# Create a user (optional, you can use the default postgres user)
CREATE USER photouser WITH ENCRYPTED PASSWORD 'your_secure_password';

# Grant privileges
GRANT ALL PRIVILEGES ON DATABASE photowebsite TO photouser;

# Exit psql
\q
```

### 3. Configure Environment Variables

Create a `.env` file in the project root (use `.env.example` as a template):

```env
USE_POSTGRES=true
DB_HOST=localhost
DB_PORT=5432
DB_USER=photouser
DB_PASSWORD=your_secure_password
DB_NAME=photowebsite
ADMIN_PASSWORD=your_admin_password
```

### 4. Install Dependencies

```bash
npm install
```

### 5. Start the Application

The database schema will be automatically created on first run:

```bash
npm start
```

## Production Deployment (Google Cloud SQL)

### 1. Create Cloud SQL Instance

```bash
gcloud sql instances create photowebsite-db \
  --database-version=POSTGRES_15 \
  --tier=db-f1-micro \
  --region=us-central1
```

### 2. Set Root Password

```bash
gcloud sql users set-password postgres \
  --instance=photowebsite-db \
  --password=YOUR_SECURE_PASSWORD
```

### 3. Create Database

```bash
gcloud sql databases create photowebsite --instance=photowebsite-db
```

### 4. Configure Cloud Run Environment Variables

When deploying to Cloud Run, set these environment variables:

```bash
USE_POSTGRES=true
DB_USER=postgres
DB_PASSWORD=your_secure_password
DB_NAME=photowebsite
INSTANCE_UNIX_SOCKET=/cloudsql/PROJECT_ID:REGION:photowebsite-db
ADMIN_PASSWORD=your_admin_password
```

### 5. Connect Cloud Run to Cloud SQL

Add the `--add-cloudsql-instances` flag when deploying:

```bash
gcloud run deploy photowebsite \
  --add-cloudsql-instances=PROJECT_ID:REGION:photowebsite-db \
  --set-env-vars="USE_POSTGRES=true,DB_NAME=photowebsite,..." \
  ...
```

## Database Schema

The application automatically creates the following tables on startup:

- `album_groups` - Groups of albums
- `albums` - Photo albums with settings
- `photos` - Individual photos linked to albums
- `site_config` - Site-wide configuration settings

All necessary indexes are also created automatically.

## Migration from SQLite

If you were previously using SQLite, you'll need to manually migrate your data:

1. Export data from SQLite
2. Transform it to PostgreSQL-compatible format
3. Import into PostgreSQL

The old SQLite database files have been removed from the repository.

## Troubleshooting

### Connection Refused

- Ensure PostgreSQL is running: `pg_isready`
- Check your connection settings in `.env`
- Verify firewall rules allow connections to port 5432

### Authentication Failed

- Double-check username and password in `.env`
- Ensure the database user has appropriate permissions

### Permission Denied

```sql
-- Grant all privileges on the database
GRANT ALL PRIVILEGES ON DATABASE photowebsite TO photouser;

-- Connect to the photowebsite database
\c photowebsite

-- Grant schema privileges
GRANT ALL ON SCHEMA public TO photouser;
```

## Useful PostgreSQL Commands

```bash
# Connect to database
psql -U photouser -d photowebsite

# List all databases
\l

# List all tables
\dt

# Describe a table
\d albums

# Run a query
SELECT * FROM albums;

# Exit psql
\q
```

## Backup and Restore

### Backup

```bash
pg_dump -U photouser photowebsite > backup.sql
```

### Restore

```bash
psql -U photouser photowebsite < backup.sql
```

For Cloud SQL:

```bash
# Export
gcloud sql export sql photowebsite-db gs://your-bucket/backup.sql \
  --database=photowebsite

# Import
gcloud sql import sql photowebsite-db gs://your-bucket/backup.sql \
  --database=photowebsite
```

