<?php

namespace App\Services\Auth;

use Firebase\JWT\JWK;
use Firebase\JWT\JWT;

class AppleJwtVerifier
{
    public function verify(string $identityToken): array
    {
        // Minimal implementation: decode payload only in local; real JWKS verify when configured
        $jwksUrl = config('services.apple.jwks_url');
        $aud = config('services.apple.client_id');
        $iss = 'https://appleid.apple.com';

        if ($jwksUrl && $aud) {
            $header = $this->readJwtHeader($identityToken);
            $keys = JWK::parseKeySet(json_decode(file_get_contents($jwksUrl), true));
            $decoded = (array) JWT::decode($identityToken, $keys);
            if (($decoded['aud'] ?? null) !== $aud) {
                throw new \RuntimeException('Invalid audience');
            }
            if (($decoded['iss'] ?? null) !== $iss) {
                throw new \RuntimeException('Invalid issuer');
            }

            return $decoded;
        }

        // local fallback: parse payload without signature verification
        $parts = explode('.', $identityToken);
        if (count($parts) < 2) {
            throw new \RuntimeException('Invalid token');
        }

        return json_decode($this->base64UrlDecode($parts[1]), true) ?: [];
    }

    private function readJwtHeader(string $jwt): array
    {
        $parts = explode('.', $jwt);

        return json_decode($this->base64UrlDecode($parts[0] ?? ''), true) ?: [];
    }

    private function base64UrlDecode(string $input): string
    {
        $remainder = strlen($input) % 4;
        if ($remainder) {
            $input .= str_repeat('=', 4 - $remainder);
        }

        return base64_decode(strtr($input, '-_', '+/')) ?: '';
    }
}
