# Dynamic Gallery URLs Fixed

## Problem
Album links in the admin panel were showing as undefined or pointing to production URLs (e.g., `https://admin.vybephoto.com/a/CODE`) even when running locally. The user expected to see `http://localhost:3000/a/CODE` in development.

## Root Cause
Two issues were causing the problem:

1. **Missing `siteConfig` variable**: All admin templates required a `siteConfig` object, but the admin route handlers weren't passing it, causing template rendering errors.

2. **Incomplete localhost detection**: The `getGalleryUrl()` function in `src/tenant.js` was checking for `localhost` and `127.0.0.1`, but not for `*.localhost` patterns like `app.localhost:3000`, which is how the multi-tenant dev environment works.

## Solution

### 1. Added `siteConfig` to All Admin Routes
Updated all admin route handlers in `src/server-multi-tenant.js` to pass a complete `siteConfig` object:

- `/admin` (main dashboard)
- `/admin/albums/:id` (album detail)
- `/admin/groups/:id` (group detail)  
- `/admin/groups/:id/delete` (delete confirmation)

Each now includes:
```javascript
siteConfig: {
  siteName: req.user.siteName || req.user.site_name || req.user.username,
  primaryColor: req.user.primaryColor || req.user.primary_color || '#3b82f6',
  primaryColorRgb: hexToRgb(req.user.primaryColor || req.user.primary_color || '#3b82f6'),
  logoUrl: req.user.logoUrl || req.user.logo_url || '/nrw-web.png',
  hotChocolateDefaultUrl: req.user.hotChocolateDefaultUrl || req.user.hot_chocolate_default_url || 'https://square.link/u/NkgDiQCk',
  hotChocolateText: req.user.hotChocolateText || req.user.hot_chocolate_text || 'Like these pics? Buy me a Hot Chocolate',
  contactEmail: req.user.contactEmail || req.user.contact_email || '',
  socialLinks: req.user.socialLinks || req.user.social_links || {},
  ageVerificationEnabled: req.user.requireAgeVerification || req.user.require_age_verification || false
}
```

### 2. Fixed `getGalleryUrl()` to Build Correct Subdomain URLs
Updated `src/tenant.js` to properly build subdomain URLs in localhost environment:

**Before:**
```javascript
if (hostname === 'localhost' || hostname === '127.0.0.1') {
  const url = `${protocol}://${req.get('host')}`;
  return url; // Returns app.localhost:3000 ❌
}
```

**After:**
```javascript
if (hostname === 'localhost' || hostname === '127.0.0.1' || hostname.endsWith('.localhost')) {
  const protocol = req.secure || req.headers['x-forwarded-proto'] === 'https' ? 'https' : 'http';
  const port = fullHost.includes(':') ? ':' + fullHost.split(':')[1] : '';
  const url = `${protocol}://${user.username}.localhost${port}`;
  return url; // Returns username.localhost:3000 ✅
}
```

This now properly:
- Detects localhost environment (including `*.localhost` patterns)
- **Builds the photographer's subdomain URL** using their username
- Preserves the port number from the original request

## How It Works

### Development Environment
When running locally and logged into `http://app.localhost:3000/admin`:
- Album links: `http://username.localhost:3000/a/SB66SRTB` ✅
- Group links: `http://username.localhost:3000/g/GROUPCODE` ✅

The URL uses the **photographer's username as the subdomain** (not "app"), matching the multi-tenant architecture where:
- `app.localhost:3000` = Admin panel (photographers manage galleries)
- `username.localhost:3000` = Public gallery (clients view photos)

### Production Environment
When deployed to production (e.g., `app.vybephoto.com`):
- For users with verified custom domain: `https://customdomain.com/a/CODE`
- For users with subdomain: `https://username.vybephoto.com/a/CODE`

## Testing

1. Start the dev server: `npm start`
2. Log in at: `http://app.localhost:3000/login` (e.g., username: "admin")
3. Go to Admin panel: `http://app.localhost:3000/admin`
4. View any album
5. The "View Album →" link should show: `http://admin.localhost:3000/a/CODE` (using your username)
6. Clicking the link opens the public gallery at the photographer's subdomain
7. When deployed to production, it will automatically use: `https://username.vybephoto.com/a/CODE`

## Files Changed

- `src/server-multi-tenant.js` - Added `siteConfig` to all admin route handlers
- `src/tenant.js` - Enhanced `getGalleryUrl()` localhost detection

## Benefits

✅ Dynamic URL generation based on environment  
✅ No hardcoded URLs  
✅ Works seamlessly in both development and production  
✅ Supports custom domains and subdomains  
✅ Proper localhost detection for multi-tenant architecture

