-- Migration: Add Subscription Tiers Configuration Table
-- This allows storing Stripe Price IDs and tier configuration in the database
-- instead of hardcoding them in environment variables

CREATE TABLE IF NOT EXISTS subscription_tiers (
  id SERIAL PRIMARY KEY,
  tier_name TEXT NOT NULL UNIQUE CHECK (tier_name IN ('starter', 'professional', 'business')),
  display_name TEXT NOT NULL,
  description TEXT,
  
  -- Pricing
  price_monthly INTEGER NOT NULL, -- Price in pence (e.g., 1000 = £10.00)
  currency TEXT NOT NULL DEFAULT 'gbp',
  
  -- Stripe Configuration
  stripe_price_id TEXT, -- Stripe Price ID (e.g., price_xxxxx)
  stripe_product_id TEXT, -- Stripe Product ID (e.g., prod_xxxxx)
  
  -- Plan Limits
  max_photos INTEGER NOT NULL,
  max_storage_bytes BIGINT NOT NULL,
  max_albums INTEGER, -- NULL = unlimited
  custom_domain_enabled BOOLEAN NOT NULL DEFAULT FALSE,
  
  -- Trial Configuration
  trial_days INTEGER NOT NULL DEFAULT 0,
  
  -- Display order and visibility
  display_order INTEGER NOT NULL DEFAULT 0,
  is_active BOOLEAN NOT NULL DEFAULT TRUE,
  is_featured BOOLEAN NOT NULL DEFAULT FALSE,
  
  -- Metadata
  created_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
  updated_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP
);

-- Create indexes
CREATE INDEX IF NOT EXISTS idx_subscription_tiers_tier_name ON subscription_tiers(tier_name);
CREATE INDEX IF NOT EXISTS idx_subscription_tiers_active ON subscription_tiers(is_active);
CREATE INDEX IF NOT EXISTS idx_subscription_tiers_stripe_price_id ON subscription_tiers(stripe_price_id);

-- Insert default tier configurations
INSERT INTO subscription_tiers (
  tier_name, display_name, description, 
  price_monthly, max_photos, max_storage_bytes, 
  max_albums, custom_domain_enabled, trial_days, display_order
) VALUES 
  (
    'starter',
    'Starter',
    '1,000 photos • 5GB storage • Subdomain only',
    1000, -- £10.00
    1000,
    5368709120, -- 5GB
    NULL, -- unlimited albums
    FALSE,
    14, -- 14-day free trial
    1
  ),
  (
    'professional',
    'Professional',
    '10,000 photos • 50GB storage • Custom domain support',
    2500, -- £25.00
    10000,
    53687091200, -- 50GB
    NULL, -- unlimited albums
    TRUE,
    0, -- no trial
    2
  ),
  (
    'business',
    'Business',
    '30,000 photos • 150GB storage • Multiple custom domains',
    7500, -- £75.00
    30000,
    161061273600, -- 150GB
    NULL, -- unlimited albums
    TRUE,
    0, -- no trial
    3
  )
ON CONFLICT (tier_name) DO NOTHING;

-- Function to update updated_at timestamp
CREATE OR REPLACE FUNCTION update_subscription_tiers_timestamp()
RETURNS TRIGGER AS $$
BEGIN
  NEW.updated_at = CURRENT_TIMESTAMP;
  RETURN NEW;
END;
$$ LANGUAGE plpgsql;

-- Create trigger for auto-updating timestamp
DROP TRIGGER IF EXISTS trigger_update_subscription_tiers_timestamp ON subscription_tiers;
CREATE TRIGGER trigger_update_subscription_tiers_timestamp
BEFORE UPDATE ON subscription_tiers
FOR EACH ROW EXECUTE FUNCTION update_subscription_tiers_timestamp();

COMMENT ON TABLE subscription_tiers IS 'Subscription tier configurations including Stripe Price IDs';
COMMENT ON COLUMN subscription_tiers.stripe_price_id IS 'Stripe Price ID for recurring subscription (e.g., price_xxxxx)';
COMMENT ON COLUMN subscription_tiers.price_monthly IS 'Monthly price in pence (1000 = £10.00)';
COMMENT ON COLUMN subscription_tiers.trial_days IS 'Number of free trial days (0 = no trial)';


