<?php

namespace App\Console\Commands;

use App\Services\PriceResolutionService;
use Illuminate\Console\Command;

class CleanupPriceResolutions extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'price-resolutions:cleanup 
                            {--days=7 : Number of days old to consider stale}
                            {--dry-run : Show what would be deleted without actually doing it}
                            {--force : Skip confirmation prompt}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Clean up stale price resolutions from the cache';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $daysOld = (int) $this->option('days');
        $dryRun = $this->option('dry-run');
        $force = $this->option('force');

        $this->info('Price Resolution Cleanup Command');
        $this->info('================================');

        $service = app(PriceResolutionService::class);
        
        // Get current stats
        $stats = $service->getCacheStats();
        $this->info("Current cache statistics:");
        $this->info("Total resolutions: {$stats['total']}");
        $this->info("Fresh resolutions: {$stats['fresh']}");
        $this->info("Stale resolutions: {$stats['stale']}");
        $this->info("Fresh percentage: {$stats['fresh_percentage']}%");

        if ($dryRun) {
            $this->warn('DRY RUN MODE - No data will be deleted');
            $this->info("Would delete resolutions older than {$daysOld} days");
            return 0;
        }

        if (!$force) {
            if (!$this->confirm("Do you want to delete resolutions older than {$daysOld} days?")) {
                $this->info('Operation cancelled.');
                return 0;
            }
        }

        $this->info("Cleaning up resolutions older than {$daysOld} days...");
        
        $deleted = $service->cleanupStaleResolutions($daysOld);
        
        $this->info("Cleanup completed! Deleted {$deleted} stale resolutions.");

        // Show updated stats
        $newStats = $service->getCacheStats();
        $this->newLine();
        $this->info('Updated cache statistics:');
        $this->info("Total resolutions: {$newStats['total']}");
        $this->info("Fresh resolutions: {$newStats['fresh']}");
        $this->info("Stale resolutions: {$newStats['stale']}");
        $this->info("Fresh percentage: {$newStats['fresh_percentage']}%");

        return 0;
    }
}
