<?php

namespace Tests\Feature;

use Tests\TestCase;
use App\Models\Price;
use App\Models\PriceResolution;
use App\Models\TotalCache;
use App\Models\Colourways;
use App\Models\ColourwayYarns;
use App\Models\Customer;
use App\Models\Suppliers;
use App\Models\StyleVersions;
use App\Models\Styles;
use App\Models\Designs;
use App\Models\Seasons;
use App\Models\Sizes;
use App\Models\Phase;
use App\Models\CustomerOrderLines;
use App\Models\CustomerOrderLineQuantities;
use App\Models\CustomerOrders;
use App\Models\ShipmentLine;
use App\Models\ShipmentLineSizes;
use App\Models\Shipment;
use App\Models\YarnColours;
use App\Models\Yarn;
use App\Services\PriceResolutionService;
use App\Services\TotalCacheService;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Support\Facades\Cache;

class CacheInvalidationTest extends TestCase
{
    use RefreshDatabase;

    protected PriceResolutionService $priceService;
    protected TotalCacheService $totalService;

    protected function setUp(): void
    {
        parent::setUp();
        $this->priceService = app(PriceResolutionService::class);
        $this->totalService = app(TotalCacheService::class);
    }

    /**
     * Test that price cache is invalidated when Price model is saved
     */
    public function test_price_saved_invalidates_price_cache()
    {
        // Create test data
        $season = Seasons::factory()->create();
        $customer = Customer::factory()->create();
        $supplier = Suppliers::factory()->create();
        $design = Designs::factory()->create();
        $style = Styles::factory()->create([
            'customers_id' => $customer->id,
            'designs_id' => $design->id,
            'seasons_id' => $season->id,
        ]);
        $styleVersion = StyleVersions::factory()->create([
            'styles_id' => $style->id,
            'factory_id' => $supplier->id,
        ]);
        $colourway = Colourways::factory()->create([
            'style_versions_id' => $styleVersion->id,
        ]);
        $price = Price::factory()->create([
            'style_versions_id' => $styleVersion->id,
            'colourways_id' => $colourway->id,
        ]);

        // Create initial price resolution
        $resolution = $this->priceService->resolve(
            $styleVersion->id,
            $colourway->id,
            null,
            null,
            $season->id
        );

        $this->assertNotNull($resolution);
        $this->assertNotNull($resolution->fresh_at);

        // Update the price
        $price->update(['cmt' => 10.50]);

        // Check that price resolution is invalidated
        $resolution->refresh();
        $this->assertNull($resolution->fresh_at);

        // Check that new resolution is created when accessed
        $newResolution = $this->priceService->resolve(
            $styleVersion->id,
            $colourway->id,
            null,
            null,
            $season->id
        );

        $this->assertNotNull($newResolution);
        $this->assertNotNull($newResolution->fresh_at);
        $this->assertEquals(10.50, $newResolution->cmt);
    }

    /**
     * Test that price cache is invalidated when Price model is deleted
     */
    public function test_price_deleted_invalidates_price_cache()
    {
        // Create test data
        $season = Seasons::factory()->create();
        $customer = Customer::factory()->create();
        $supplier = Suppliers::factory()->create();
        $design = Designs::factory()->create();
        $style = Styles::factory()->create([
            'customers_id' => $customer->id,
            'designs_id' => $design->id,
            'seasons_id' => $season->id,
        ]);
        $styleVersion = StyleVersions::factory()->create([
            'styles_id' => $style->id,
            'factory_id' => $supplier->id,
        ]);
        $colourway = Colourways::factory()->create([
            'style_versions_id' => $styleVersion->id,
        ]);
        $price = Price::factory()->create([
            'style_versions_id' => $styleVersion->id,
            'colourways_id' => $colourway->id,
        ]);

        // Create initial price resolution
        $resolution = $this->priceService->resolve(
            $styleVersion->id,
            $colourway->id,
            null,
            null,
            $season->id
        );

        $this->assertNotNull($resolution);
        $this->assertNotNull($resolution->fresh_at);

        // Delete the price
        $price->delete();

        // Check that price resolution is invalidated
        $resolution->refresh();
        $this->assertNull($resolution->fresh_at);
    }

    /**
     * Test that price cache is invalidated when Colourways model is updated
     */
    public function test_colourway_updated_invalidates_price_cache()
    {
        // Create test data
        $season = Seasons::factory()->create();
        $customer = Customer::factory()->create();
        $supplier = Suppliers::factory()->create();
        $design = Designs::factory()->create();
        $style = Styles::factory()->create([
            'customers_id' => $customer->id,
            'designs_id' => $design->id,
            'seasons_id' => $season->id,
        ]);
        $styleVersion = StyleVersions::factory()->create([
            'styles_id' => $style->id,
            'factory_id' => $supplier->id,
        ]);
        $colourway = Colourways::factory()->create([
            'style_versions_id' => $styleVersion->id,
        ]);
        $price = Price::factory()->create([
            'style_versions_id' => $styleVersion->id,
            'colourways_id' => $colourway->id,
        ]);

        // Create initial price resolution
        $resolution = $this->priceService->resolve(
            $styleVersion->id,
            $colourway->id,
            null,
            null,
            $season->id
        );

        $this->assertNotNull($resolution);
        $this->assertNotNull($resolution->fresh_at);

        // Update the colourway
        $colourway->update(['name' => 'Updated Colourway']);

        // Check that price resolution is invalidated
        $resolution->refresh();
        $this->assertNull($resolution->fresh_at);
    }

    /**
     * Test that price cache is invalidated when ColourwayYarns model is saved
     */
    public function test_colourway_yarn_saved_invalidates_price_cache()
    {
        // Create test data
        $season = Seasons::factory()->create();
        $customer = Customer::factory()->create();
        $supplier = Suppliers::factory()->create();
        $design = Designs::factory()->create();
        $style = Styles::factory()->create([
            'customers_id' => $customer->id,
            'designs_id' => $design->id,
            'seasons_id' => $season->id,
        ]);
        $styleVersion = StyleVersions::factory()->create([
            'styles_id' => $style->id,
            'factory_id' => $supplier->id,
        ]);
        $colourway = Colourways::factory()->create([
            'style_versions_id' => $styleVersion->id,
        ]);
        $price = Price::factory()->create([
            'style_versions_id' => $styleVersion->id,
            'colourways_id' => $colourway->id,
        ]);

        // Create initial price resolution
        $resolution = $this->priceService->resolve(
            $styleVersion->id,
            $colourway->id,
            null,
            null,
            $season->id
        );

        $this->assertNotNull($resolution);
        $this->assertNotNull($resolution->fresh_at);

        // Create yarn and yarn colour
        $yarn = Yarn::factory()->create();
        $yarnColour = YarnColours::factory()->create([
            'yarn_id' => $yarn->id,
        ]);

        // Create colourway yarn
        ColourwayYarns::factory()->create([
            'colourways_id' => $colourway->id,
            'yarn_colours_id' => $yarnColour->id,
        ]);

        // Check that price resolution is invalidated
        $resolution->refresh();
        $this->assertNull($resolution->fresh_at);
    }

    /**
     * Test that price cache is invalidated when Customer model is updated
     */
    public function test_customer_updated_invalidates_price_cache()
    {
        // Create test data
        $season = Seasons::factory()->create();
        $customer = Customer::factory()->create();
        $supplier = Suppliers::factory()->create();
        $design = Designs::factory()->create();
        $style = Styles::factory()->create([
            'customers_id' => $customer->id,
            'designs_id' => $design->id,
            'seasons_id' => $season->id,
        ]);
        $styleVersion = StyleVersions::factory()->create([
            'styles_id' => $style->id,
            'factory_id' => $supplier->id,
        ]);
        $colourway = Colourways::factory()->create([
            'style_versions_id' => $styleVersion->id,
        ]);
        $price = Price::factory()->create([
            'style_versions_id' => $styleVersion->id,
            'colourways_id' => $colourway->id,
        ]);

        // Create initial price resolution
        $resolution = $this->priceService->resolve(
            $styleVersion->id,
            $colourway->id,
            null,
            null,
            $season->id
        );

        $this->assertNotNull($resolution);
        $this->assertNotNull($resolution->fresh_at);

        // Update the customer
        $customer->update(['name' => 'Updated Customer']);

        // Check that price resolution is invalidated
        $resolution->refresh();
        $this->assertNull($resolution->fresh_at);
    }

    /**
     * Test that price cache is invalidated when Supplier model is updated
     */
    public function test_supplier_updated_invalidates_price_cache()
    {
        // Create test data
        $season = Seasons::factory()->create();
        $customer = Customer::factory()->create();
        $supplier = Suppliers::factory()->create();
        $design = Designs::factory()->create();
        $style = Styles::factory()->create([
            'customers_id' => $customer->id,
            'designs_id' => $design->id,
            'seasons_id' => $season->id,
        ]);
        $styleVersion = StyleVersions::factory()->create([
            'styles_id' => $style->id,
            'factory_id' => $supplier->id,
        ]);
        $colourway = Colourways::factory()->create([
            'style_versions_id' => $styleVersion->id,
        ]);
        $price = Price::factory()->create([
            'style_versions_id' => $styleVersion->id,
            'colourways_id' => $colourway->id,
        ]);

        // Create initial price resolution
        $resolution = $this->priceService->resolve(
            $styleVersion->id,
            $colourway->id,
            null,
            null,
            $season->id
        );

        $this->assertNotNull($resolution);
        $this->assertNotNull($resolution->fresh_at);

        // Update the supplier
        $supplier->update(['name' => 'Updated Supplier']);

        // Check that price resolution is invalidated
        $resolution->refresh();
        $this->assertNull($resolution->fresh_at);
    }

    /**
     * Test that price cache is invalidated when StyleVersions model is updated with relevant fields
     */
    public function test_style_version_updated_invalidates_price_cache()
    {
        // Create test data
        $season = Seasons::factory()->create();
        $customer = Customer::factory()->create();
        $supplier = Suppliers::factory()->create();
        $design = Designs::factory()->create();
        $style = Styles::factory()->create([
            'customers_id' => $customer->id,
            'designs_id' => $design->id,
            'seasons_id' => $season->id,
        ]);
        $styleVersion = StyleVersions::factory()->create([
            'styles_id' => $style->id,
            'factory_id' => $supplier->id,
        ]);
        $colourway = Colourways::factory()->create([
            'style_versions_id' => $styleVersion->id,
        ]);
        $price = Price::factory()->create([
            'style_versions_id' => $styleVersion->id,
            'colourways_id' => $colourway->id,
        ]);

        // Create initial price resolution
        $resolution = $this->priceService->resolve(
            $styleVersion->id,
            $colourway->id,
            null,
            null,
            $season->id
        );

        $this->assertNotNull($resolution);
        $this->assertNotNull($resolution->fresh_at);

        // Update the style version with relevant fields
        $styleVersion->update(['factory_id' => Suppliers::factory()->create()->id]);

        // Check that price resolution is invalidated
        $resolution->refresh();
        $this->assertNull($resolution->fresh_at);
    }

    /**
     * Test that total cache is invalidated when CustomerOrderLines model is updated
     */
    public function test_customer_order_line_updated_invalidates_total_cache()
    {
        // Create test data
        $season = Seasons::factory()->create();
        $customer = Customer::factory()->create();
        $supplier = Suppliers::factory()->create();
        $design = Designs::factory()->create();
        $style = Styles::factory()->create([
            'customers_id' => $customer->id,
            'designs_id' => $design->id,
            'seasons_id' => $season->id,
        ]);
        $styleVersion = StyleVersions::factory()->create([
            'styles_id' => $style->id,
            'factory_id' => $supplier->id,
        ]);
        $colourway = Colourways::factory()->create([
            'style_versions_id' => $styleVersion->id,
        ]);
        $price = Price::factory()->create([
            'style_versions_id' => $styleVersion->id,
            'colourways_id' => $colourway->id,
        ]);
        $customerOrder = CustomerOrders::factory()->create([
            'customers_id' => $customer->id,
        ]);
        $customerOrderLine = CustomerOrderLines::factory()->create([
            'customer_orders_id' => $customerOrder->id,
            'colourways_id' => $colourway->id,
        ]);

        // Warm up total cache
        $this->totalService->warmupCustomerOrderLine($customerOrderLine);
        $this->totalService->warmupCustomerOrder($customerOrder);

        // Verify cache exists
        $lineCache = $this->totalService->get('customer_order_line', $customerOrderLine->id, 'prices');
        $orderCache = $this->totalService->get('customer_order', $customerOrder->id, 'prices');
        $this->assertNotNull($lineCache);
        $this->assertNotNull($orderCache);

        // Update the customer order line
        $customerOrderLine->update(['qty' => 100]);

        // Check that total cache is invalidated
        $lineCacheAfter = $this->totalService->get('customer_order_line', $customerOrderLine->id, 'prices');
        $orderCacheAfter = $this->totalService->get('customer_order', $customerOrder->id, 'prices');
        $this->assertNull($lineCacheAfter);
        $this->assertNull($orderCacheAfter);
    }

    /**
     * Test that total cache is invalidated when CustomerOrderLineQuantities model is updated
     */
    public function test_customer_order_line_quantities_updated_invalidates_total_cache()
    {
        // Create test data
        $season = Seasons::factory()->create();
        $customer = Customer::factory()->create();
        $supplier = Suppliers::factory()->create();
        $design = Designs::factory()->create();
        $style = Styles::factory()->create([
            'customers_id' => $customer->id,
            'designs_id' => $design->id,
            'seasons_id' => $season->id,
        ]);
        $styleVersion = StyleVersions::factory()->create([
            'styles_id' => $style->id,
            'factory_id' => $supplier->id,
        ]);
        $colourway = Colourways::factory()->create([
            'style_versions_id' => $styleVersion->id,
        ]);
        $price = Price::factory()->create([
            'style_versions_id' => $styleVersion->id,
            'colourways_id' => $colourway->id,
        ]);
        $customerOrder = CustomerOrders::factory()->create([
            'customers_id' => $customer->id,
        ]);
        $customerOrderLine = CustomerOrderLines::factory()->create([
            'customer_orders_id' => $customerOrder->id,
            'colourways_id' => $colourway->id,
        ]);
        $size = Sizes::factory()->create();
        $customerOrderLineQuantity = CustomerOrderLineQuantities::factory()->create([
            'customer_order_lines_id' => $customerOrderLine->id,
            'sizes_id' => $size->id,
        ]);

        // Warm up total cache
        $this->totalService->warmupCustomerOrderLine($customerOrderLine);
        $this->totalService->warmupCustomerOrder($customerOrder);

        // Verify cache exists
        $lineCache = $this->totalService->get('customer_order_line', $customerOrderLine->id, 'prices');
        $orderCache = $this->totalService->get('customer_order', $customerOrder->id, 'prices');
        $this->assertNotNull($lineCache);
        $this->assertNotNull($orderCache);

        // Update the customer order line quantity
        $customerOrderLineQuantity->update(['qty' => 50]);

        // Check that total cache is invalidated
        $lineCacheAfter = $this->totalService->get('customer_order_line', $customerOrderLine->id, 'prices');
        $orderCacheAfter = $this->totalService->get('customer_order', $customerOrder->id, 'prices');
        $this->assertNull($lineCacheAfter);
        $this->assertNull($orderCacheAfter);
    }

    /**
     * Test that total cache is invalidated when ShipmentLine model is updated
     */
    public function test_shipment_line_updated_invalidates_total_cache()
    {
        // Create test data
        $season = Seasons::factory()->create();
        $customer = Customer::factory()->create();
        $supplier = Suppliers::factory()->create();
        $design = Designs::factory()->create();
        $style = Styles::factory()->create([
            'customers_id' => $customer->id,
            'designs_id' => $design->id,
            'seasons_id' => $season->id,
        ]);
        $styleVersion = StyleVersions::factory()->create([
            'styles_id' => $style->id,
            'factory_id' => $supplier->id,
        ]);
        $colourway = Colourways::factory()->create([
            'style_versions_id' => $styleVersion->id,
        ]);
        $price = Price::factory()->create([
            'style_versions_id' => $styleVersion->id,
            'colourways_id' => $colourway->id,
        ]);
        $customerOrder = CustomerOrders::factory()->create([
            'customers_id' => $customer->id,
        ]);
        $customerOrderLine = CustomerOrderLines::factory()->create([
            'customer_orders_id' => $customerOrder->id,
            'colourways_id' => $colourway->id,
        ]);
        $shipment = Shipment::factory()->create();
        $shipmentLine = ShipmentLine::factory()->create([
            'shipment_id' => $shipment->id,
            'customer_order_lines_id' => $customerOrderLine->id,
        ]);

        // Warm up total cache
        $this->totalService->warmupShipmentLine($shipmentLine);
        $this->totalService->warmupShipment($shipment);

        // Verify cache exists
        $lineCache = $this->totalService->get('shipment_line', $shipmentLine->id, 'prices');
        $shipmentCache = $this->totalService->get('shipment', $shipment->id, 'prices');
        $this->assertNotNull($lineCache);
        $this->assertNotNull($shipmentCache);

        // Update the shipment line
        $shipmentLine->update(['qty' => 75]);

        // Check that total cache is invalidated
        $lineCacheAfter = $this->totalService->get('shipment_line', $shipmentLine->id, 'prices');
        $shipmentCacheAfter = $this->totalService->get('shipment', $shipment->id, 'prices');
        $this->assertNull($lineCacheAfter);
        $this->assertNull($shipmentCacheAfter);
    }

    /**
     * Test that total cache is invalidated when ShipmentLineSizes model is updated
     */
    public function test_shipment_line_sizes_updated_invalidates_total_cache()
    {
        // Create test data
        $season = Seasons::factory()->create();
        $customer = Customer::factory()->create();
        $supplier = Suppliers::factory()->create();
        $design = Designs::factory()->create();
        $style = Styles::factory()->create([
            'customers_id' => $customer->id,
            'designs_id' => $design->id,
            'seasons_id' => $season->id,
        ]);
        $styleVersion = StyleVersions::factory()->create([
            'styles_id' => $style->id,
            'factory_id' => $supplier->id,
        ]);
        $colourway = Colourways::factory()->create([
            'style_versions_id' => $styleVersion->id,
        ]);
        $price = Price::factory()->create([
            'style_versions_id' => $styleVersion->id,
            'colourways_id' => $colourway->id,
        ]);
        $customerOrder = CustomerOrders::factory()->create([
            'customers_id' => $customer->id,
        ]);
        $customerOrderLine = CustomerOrderLines::factory()->create([
            'customer_orders_id' => $customerOrder->id,
            'colourways_id' => $colourway->id,
        ]);
        $shipment = Shipment::factory()->create();
        $shipmentLine = ShipmentLine::factory()->create([
            'shipment_id' => $shipment->id,
            'customer_order_lines_id' => $customerOrderLine->id,
        ]);
        $size = Sizes::factory()->create();
        $shipmentLineSize = ShipmentLineSizes::factory()->create([
            'shipment_lines_id' => $shipmentLine->id,
            'sizes_id' => $size->id,
        ]);

        // Warm up total cache
        $this->totalService->warmupShipmentLine($shipmentLine);
        $this->totalService->warmupShipment($shipment);

        // Verify cache exists
        $lineCache = $this->totalService->get('shipment_line', $shipmentLine->id, 'prices');
        $shipmentCache = $this->totalService->get('shipment', $shipment->id, 'prices');
        $this->assertNotNull($lineCache);
        $this->assertNotNull($shipmentCache);

        // Update the shipment line size
        $shipmentLineSize->update(['qty' => 25]);

        // Check that total cache is invalidated
        $lineCacheAfter = $this->totalService->get('shipment_line', $shipmentLine->id, 'prices');
        $shipmentCacheAfter = $this->totalService->get('shipment', $shipment->id, 'prices');
        $this->assertNull($lineCacheAfter);
        $this->assertNull($shipmentCacheAfter);
    }

    /**
     * Test that cache invalidation works for general prices (NULL colourway)
     */
    public function test_general_price_invalidates_all_colourway_caches()
    {
        // Create test data
        $season = Seasons::factory()->create();
        $customer = Customer::factory()->create();
        $supplier = Suppliers::factory()->create();
        $design = Designs::factory()->create();
        $style = Styles::factory()->create([
            'customers_id' => $customer->id,
            'designs_id' => $design->id,
            'seasons_id' => $season->id,
        ]);
        $styleVersion = StyleVersions::factory()->create([
            'styles_id' => $style->id,
            'factory_id' => $supplier->id,
        ]);
        
        // Create multiple colourways
        $colourway1 = Colourways::factory()->create([
            'style_versions_id' => $styleVersion->id,
        ]);
        $colourway2 = Colourways::factory()->create([
            'style_versions_id' => $styleVersion->id,
        ]);

        // Create general price (NULL colourway)
        $generalPrice = Price::factory()->create([
            'style_versions_id' => $styleVersion->id,
            'colourways_id' => null,
        ]);

        // Create price resolutions for both colourways
        $resolution1 = $this->priceService->resolve(
            $styleVersion->id,
            $colourway1->id,
            null,
            null,
            $season->id
        );
        $resolution2 = $this->priceService->resolve(
            $styleVersion->id,
            $colourway2->id,
            null,
            null,
            $season->id
        );

        $this->assertNotNull($resolution1);
        $this->assertNotNull($resolution2);
        $this->assertNotNull($resolution1->fresh_at);
        $this->assertNotNull($resolution2->fresh_at);

        // Update the general price
        $generalPrice->update(['cmt' => 15.75]);

        // Check that both price resolutions are invalidated
        $resolution1->refresh();
        $resolution2->refresh();
        $this->assertNull($resolution1->fresh_at);
        $this->assertNull($resolution2->fresh_at);
    }

    /**
     * Test that cache invalidation cascades properly through relationships
     */
    public function test_cache_invalidation_cascades_through_relationships()
    {
        // Create test data
        $season = Seasons::factory()->create();
        $customer = Customer::factory()->create();
        $supplier = Suppliers::factory()->create();
        $design = Designs::factory()->create();
        $style = Styles::factory()->create([
            'customers_id' => $customer->id,
            'designs_id' => $design->id,
            'seasons_id' => $season->id,
        ]);
        $styleVersion = StyleVersions::factory()->create([
            'styles_id' => $style->id,
            'factory_id' => $supplier->id,
        ]);
        $colourway = Colourways::factory()->create([
            'style_versions_id' => $styleVersion->id,
        ]);
        $price = Price::factory()->create([
            'style_versions_id' => $styleVersion->id,
            'colourways_id' => $colourway->id,
        ]);
        $customerOrder = CustomerOrders::factory()->create([
            'customers_id' => $customer->id,
        ]);
        $customerOrderLine = CustomerOrderLines::factory()->create([
            'customer_orders_id' => $customerOrder->id,
            'colourways_id' => $colourway->id,
        ]);
        $shipment = Shipment::factory()->create();
        $shipmentLine = ShipmentLine::factory()->create([
            'shipment_id' => $shipment->id,
            'customer_order_lines_id' => $customerOrderLine->id,
        ]);

        // Warm up all caches
        $this->totalService->warmupCustomerOrderLine($customerOrderLine);
        $this->totalService->warmupCustomerOrder($customerOrder);
        $this->totalService->warmupShipmentLine($shipmentLine);
        $this->totalService->warmupShipment($shipment);

        // Verify all caches exist
        $lineCache = $this->totalService->get('customer_order_line', $customerOrderLine->id, 'prices');
        $orderCache = $this->totalService->get('customer_order', $customerOrder->id, 'prices');
        $shipmentLineCache = $this->totalService->get('shipment_line', $shipmentLine->id, 'prices');
        $shipmentCache = $this->totalService->get('shipment', $shipment->id, 'prices');
        
        $this->assertNotNull($lineCache);
        $this->assertNotNull($orderCache);
        $this->assertNotNull($shipmentLineCache);
        $this->assertNotNull($shipmentCache);

        // Update the price (should invalidate all related caches)
        $price->update(['cmt' => 20.00]);

        // Check that all caches are invalidated
        $lineCacheAfter = $this->totalService->get('customer_order_line', $customerOrderLine->id, 'prices');
        $orderCacheAfter = $this->totalService->get('customer_order', $customerOrder->id, 'prices');
        $shipmentLineCacheAfter = $this->totalService->get('shipment_line', $shipmentLine->id, 'prices');
        $shipmentCacheAfter = $this->totalService->get('shipment', $shipment->id, 'prices');
        
        $this->assertNull($lineCacheAfter);
        $this->assertNull($orderCacheAfter);
        $this->assertNull($shipmentLineCacheAfter);
        $this->assertNull($shipmentCacheAfter);
    }

    /**
     * Test that cache warming works correctly after invalidation
     */
    public function test_cache_warming_after_invalidation()
    {
        // Create test data
        $season = Seasons::factory()->create();
        $customer = Customer::factory()->create();
        $supplier = Suppliers::factory()->create();
        $design = Designs::factory()->create();
        $style = Styles::factory()->create([
            'customers_id' => $customer->id,
            'designs_id' => $design->id,
            'seasons_id' => $season->id,
        ]);
        $styleVersion = StyleVersions::factory()->create([
            'styles_id' => $style->id,
            'factory_id' => $supplier->id,
        ]);
        $colourway = Colourways::factory()->create([
            'style_versions_id' => $styleVersion->id,
        ]);
        $price = Price::factory()->create([
            'style_versions_id' => $styleVersion->id,
            'colourways_id' => $colourway->id,
        ]);

        // Create initial price resolution
        $resolution = $this->priceService->resolve(
            $styleVersion->id,
            $colourway->id,
            null,
            null,
            $season->id
        );

        $this->assertNotNull($resolution);
        $this->assertNotNull($resolution->fresh_at);

        // Update the price to invalidate cache
        $price->update(['cmt' => 25.50]);

        // Check that resolution is invalidated
        $resolution->refresh();
        $this->assertNull($resolution->fresh_at);

        // Access the resolution again (should trigger warming)
        $newResolution = $this->priceService->resolve(
            $styleVersion->id,
            $colourway->id,
            null,
            null,
            $season->id
        );

        // Check that new resolution is created and warmed up
        $this->assertNotNull($newResolution);
        $this->assertNotNull($newResolution->fresh_at);
        $this->assertEquals(25.50, $newResolution->cmt);
    }
}
