<?php

namespace App\Jobs;

use App\Models\Receipt;
use App\Services\Matching\ReceiptMatchingService;
use Illuminate\Bus\Batchable;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;

class MatchReceipt implements ShouldQueue
{
    use Batchable, Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    public int $tries = 3;
    public int $timeout = 60; // Reduced timeout for faster processing
    public int $maxExceptions = 2; // Fail faster on repeated exceptions

    public function __construct(
        public int $receiptId,
        public ?int $userId = null
    ) {}

    public function backoff(): array
    {
        return [15, 60, 300];
    }

    public function retryUntil(): \DateTimeInterface
    {
        return now()->addHours(6);
    }

    public function handle(ReceiptMatchingService $matchingService): void
    {
        try {
            $receipt = Receipt::find($this->receiptId);
            if (!$receipt) return;

            $this->userId = $this->userId ?? $receipt->user_id;
            
            $matchingService->attemptMatchForReceipt($receipt);
        } catch (\Throwable $e) {
            \Log::warning('MatchReceipt: transient failure, will retry', [
                'receipt_id' => $this->receiptId,
                'error' => $e->getMessage(),
                'attempt' => method_exists($this, 'attempts') ? $this->attempts() : null,
            ]);
            $schedule = (array) $this->backoff();
            $index = max(0, min((int)($this->attempts() - 1), count($schedule) - 1));
            $this->release((int) $schedule[$index]);
        }
    }
}


