const db = require('./src/db-multi-tenant');
require('dotenv').config();

async function checkStripeConfig() {
  try {
    console.log('========================================');
    console.log('Checking Stripe Configuration');
    console.log('========================================\n');
    
    // Check environment variables
    console.log('1. Environment Variables:');
    console.log(`   STRIPE_SECRET_KEY: ${process.env.STRIPE_SECRET_KEY ? '✅ Set (length: ' + process.env.STRIPE_SECRET_KEY.length + ')' : '❌ NOT SET'}`);
    console.log(`   STRIPE_WEBHOOK_SECRET: ${process.env.STRIPE_WEBHOOK_SECRET ? '✅ Set' : '⚠️  Not set (optional)'}`);
    
    // Check database Price IDs
    console.log('\n2. Database Price IDs:');
    const result = await db.pool.query(`
      SELECT tier_key, tier_name, stripe_price_id, stripe_product_id 
      FROM subscription_tiers 
      ORDER BY display_order;
    `);
    
    let allConfigured = true;
    result.rows.forEach(tier => {
      const hasPriceId = tier.stripe_price_id && tier.stripe_price_id.trim() !== '';
      console.log(`   ${tier.tier_name} (${tier.tier_key}):`);
      console.log(`      Price ID: ${hasPriceId ? '✅ ' + tier.stripe_price_id : '❌ NOT SET'}`);
      console.log(`      Product ID: ${tier.stripe_product_id || '(not set)'}`);
      
      if (!hasPriceId) {
        allConfigured = false;
      }
    });
    
    console.log('\n3. Overall Status:');
    const stripeKeySet = !!process.env.STRIPE_SECRET_KEY;
    
    if (stripeKeySet && allConfigured) {
      console.log('   ✅ Stripe is fully configured!');
    } else {
      console.log('   ❌ Stripe configuration is incomplete:');
      if (!stripeKeySet) {
        console.log('      - STRIPE_SECRET_KEY is missing from .env file');
      }
      if (!allConfigured) {
        console.log('      - Some tiers are missing Stripe Price IDs in the database');
      }
    }
    
    console.log('\n========================================');
    console.log('Next Steps:');
    console.log('========================================');
    
    if (!stripeKeySet) {
      console.log('1. Get your Stripe Secret Key from: https://dashboard.stripe.com/test/apikeys');
      console.log('2. Add it to your .env file:');
      console.log('   STRIPE_SECRET_KEY=sk_test_...');
    }
    
    if (!allConfigured) {
      console.log('1. Go to: http://app.localhost:3000/superuser/tiers');
      console.log('2. Edit each tier and add the Stripe Price ID');
      console.log('3. Get Price IDs from: https://dashboard.stripe.com/test/products');
    }
    
  } catch (error) {
    console.error('Error checking configuration:', error);
  } finally {
    process.exit(0);
  }
}

checkStripeConfig();

