<?php

namespace App\Http\Livewire\Sales\Pricing;

use App\Models\Price;
use App\Models\Sizes;
use App\Models\Styles;
use Livewire\Component;
use App\Helper\Functions;
use App\Models\Suppliers;
use Livewire\Attributes\On;
use App\Models\StyleVersions;
use Livewire\Attributes\Computed;
use Illuminate\Support\Facades\DB;
use App\Http\Livewire\BaseComponent;
use Illuminate\Support\Facades\Gate;
use App\Models\CustomerOrderLineQuantities;

class KwaPriceRow extends BaseComponent
{
    public $style, $styleId, $printList = [];

    public function mount($style)
    {
        $this->styleId = $style->first()->first()->style_id;
        $this->style = json_decode(json_encode($style->toArray()), true);

        foreach($this->style as $sv){
            foreach($sv as $price){
                if(!$price['id']){
                    $this->createPrice($price['style_version_id']);
                }
            }
        }
    }

    public function updated($propertyName)
    {
        $this->validateOnly($propertyName);

        $propertyParts = explode('.', $propertyName);

        if (count($propertyParts) < 4) {
            return; // Invalid property
        }

        list($arrayName, $versionId, $priceIndex, $field) = $propertyParts;

        if ($arrayName !== 'style') {
            return; // We are only handling 'style' updates
        }

        $newValue = data_get($this->style, "$versionId.$priceIndex.$field");

        if (in_array($field, $this->getStyleVersionFields())) {
            $this->updateStyleVersion($versionId, $field, $newValue);
        } elseif (in_array($field, $this->getPriceFields())) {
            $this->updatePrice($versionId, $priceIndex, $field, $newValue);
        }
    }

    protected function getStyleVersionFields()
    {
        return [
            'style_version_name',
            'factory_id',
        ];
    }

    protected function getPriceFields()
    {
        return [
            'price_name',
            'model',
            'sizes_id',
            'colourways_id',
            'colour_type',
            'yarnkg',
            'yarnkg_currency',
            'yarn_trans',
            'yarn_trans_currency',
            'weight',
            'costed_weight',
            'cmt',
            'cmt_status',
            'valid_until',
            'aoc',
            'accessory',
            'accessory_currency',
            'label',
            'label_currency',
            'gmt_trans',
            'gmt_trans_currency',
            'testing',
            'testing_courier',
            'embroidery',
            'embroidery_currency',
            'duty',
            'quote',
            'quote_status',
            'target',
            'min_qty',
            'notes',
            'customer_notes',
        ];
    }

    protected $rules = [
        'printList.*' => 'nullable|boolean',

        // StyleVersion fields
        'style.*.*.style_version_name' => 'required|string|max:255',
        'style.*.*.factory_id' => 'required|integer|exists:suppliers,id',

        // Price fields
        'style.*.*.price_name' => 'nullable|string|max:255',
        'style.*.*.model' => 'required|string|in:standard,full_factored,manual',
        'style.*.*.sizes_id' => 'required|integer|exists:sizes,id',
        'style.*.*.colourways_id' => 'nullable|integer',
        'style.*.*.colour_type' => 'nullable|string|in:solid,melange,mixed',
        'style.*.*.yarnkg' => 'nullable|numeric|min:0',
        'style.*.*.yarnkg_currency' => 'nullable|string|in:£,€,',
        'style.*.*.yarn_trans' => 'nullable|numeric|min:0',
        'style.*.*.yarn_trans_currency' => 'nullable|string|in:£,€,',
        'style.*.*.weight' => 'required|numeric|min:0',
        'style.*.*.costed_weight' => 'required|numeric|min:0',
        'style.*.*.cmt' => 'nullable|numeric|min:0',
        'style.*.*.cmt_status' => 'nullable|string|in:not_confirmed,confirmed,estimate',
        'style.*.*.valid_until' => 'nullable|date',
        'style.*.*.aoc' => 'nullable|numeric|min:0|max:100',
        'style.*.*.accessory' => 'nullable|numeric|min:0',
        'style.*.*.accessory_currency' => 'nullable|string|in:£,€,',
        'style.*.*.label' => 'nullable|numeric|min:0',
        'style.*.*.label_currency' => 'nullable|string|in:£,€,',
        'style.*.*.gmt_trans' => 'nullable|numeric|min:0',
        'style.*.*.gmt_trans_currency' => 'nullable|string|in:£,€,',
        'style.*.*.testing' => 'nullable|numeric|min:0',
        'style.*.*.testing_courier' => 'nullable|numeric|min:0',
        'style.*.*.embroidery' => 'nullable|numeric|min:0',
        'style.*.*.embroidery_currency' => 'nullable|string|in:£,€,',
        'style.*.*.duty' => 'nullable|numeric|min:0|max:100',
        'style.*.*.quote' => 'nullable|numeric|min:0',
        'style.*.*.quote_status' => 'nullable|string|in:not_confirmed,confirmed,estimate',
        'style.*.*.target' => 'nullable|numeric|min:0',
        'style.*.*.min_qty' => 'nullable|integer|min:0',
        'style.*.*.notes' => 'nullable|string|max:5000',
        'style.*.*.customer_notes' => 'nullable|string|max:5000',
    ];

    protected function updateStyleVersion($versionId, $field, $value)
    {
        $styleVersion = StyleVersions::with('styles.seasons')->find($versionId);

        if ($styleVersion && Gate::allows('price:update') && !$styleVersion->styles->seasons->locked) {
            $styleVersion->$field = $value;
            $styleVersion->save();
        }
    }

    protected function updatePrice($versionId, $priceIndex, $field, $value)
    {
        // dd($this->style[$versionId][$priceIndex][$field], $value);
        $priceId = $this->style[$versionId][$priceIndex]['id'];
        $price = Price::with('style_versions.styles.seasons')->find($priceId);

        if ($price && Gate::allows('price:update') && !$price->style_versions->styles->seasons->locked) {
            $price->$field = $value;
            $price->save();
        }

        if($field == 'weight' && (empty($this->style[$versionId][$priceIndex]['costed_weight']) || $this->style[$versionId][$priceIndex]['costed_weight'] == 0)){
            $price->costed_weight = $value * 1.1;
            $this->style[$versionId][$priceIndex]['costed_weight'] = $value * 1.1;
            $price->save();
        }


        $colq = CustomerOrderLineQuantities::
        with('customer_order_lines.colourways.style_versions.styles.seasons', 'customer_order_lines.colourways.style_versions.prices')->
        whereRelation('customer_order_lines.colourways', 'style_versions_id', $versionId)->
        get();
        // dd($colq->pluck('id'), $this->styles[$p]['style_version_id']);

        foreach($colq as $line){
            $price = Functions::getPrice(
                $line->customer_order_lines->colourways->style_versions->prices,
                $line->sizes_id,
                $line->customer_order_lines->colourways->colour_type,
                $line->customer_order_lines->colourways->id
            );

            // dd($price, $line->sizes_id, $line->customer_order_lines->colourways->style_versions->prices);

            // dd($price->cost,  $price->id, $priceId);

            if(!is_string($price)){
                // dd($price);
                $line->update([
                    'prices_id' => $price->id,

                    // Price resolution system will handle caching automatically
                    // No need to set old cache columns
                ]);
            }
        }
    }

    public function render()
    {
        // $this->loadPrice();
        return view('livewire.sales.pricing.kwa-price-row');
    }

    public function countPricesInVersion($version){
        return count($version);
    }

    public function countPricesInStyle($versions){
        return array_reduce($versions, function ($carry, $versionPrices) {
            return $carry + count($versionPrices);
        }, 0);
    }

    #[On('select-all')]
    public function selectAll(){
        foreach($this->style as $sv){
            foreach($sv as $price){
                $this->printList[$price['id']] = TRUE;
                $this->dispatch('toggle-selected-price', id: $price['id']);
            }
        }
    }

    #[Computed(cache: true, key: 'factories-with-countries')]
    public function factories(){
        return Suppliers::with('countries')->where('type', 'factory')->get();
    }
    #[Computed]
    public function sizes(){
        return Sizes::get();
    }

    public function clonePrice($price){
        Gate::authorize('price:update');
        $price = Price::find($price)->duplicate();
        $price->cmt_status = FALSE;
        $price->quote_status = FALSE;
        $price->cmt_status = 'not_confirmed';
        $price->quote_status = 'not_confirmed';
        $price->save();
        $this->loadPrice();
    }
    public function createPrice($version){
        Gate::authorize('price:update');
        $price = new Price;
        $price->style_versions_id = $version;
        $price->model = 'standard';
        $price->cmt_status = 'not_confirmed';
        $price->quote_status = 'not_confirmed';
        $price->sizes_id = '1';
        $price->save();
        $this->loadPrice();
    }
    public function deletePrice($price){
        Gate::authorize('price:update');
        $price = Price::find($price);
        $price->delete();
        $this->loadPrice();
    }

    public function loadPrice(){

        $where = [];
        $where[] = "\nwhere 1 ";
        $where[] = "\nand style_versions.deleted_at IS NULL ";
        $where[] = "\nand styles.cancelled != 1";
        $where[] = "\nand styles.id =" . $this->styleId;

        $query = "
            select
                prices.id,
                prices.yarn AS yarnkg,
                prices.yarn_currency AS yarnkg_currency,
                designs.id AS design_id,
                designs.description AS description,
                styles.id AS style_id,
                styles.customer_ref AS customer_ref,
                styles.seasons_id AS seasons_id,
                seasons.euro_rate AS euro_rate,
                seasons.usd_rate AS usd_rate,
                style_versions.id AS style_version_id,
                style_versions.name AS style_version_name,
                suppliers.id AS factory_id,
                suppliers.name AS factory,
                prices.id AS price_id,
                prices.name AS price_name,
                prices.model AS model,
                prices.comments_toggle AS comments_toggle,
                prices.sizes_id AS sizes_id,
                prices.colour_type AS colour_type,
                prices.colourways_id AS colourways_id,
                prices.yarn_trans_currency AS yarn_trans_currency,
                format(prices.yarn_trans, 2) AS yarn_trans,
                format(prices.weight, 3) AS weight,
                format(prices.costed_weight, 3) AS costed_weight,
                format(prices.cmt, 2) AS cmt,
                prices.cmt_status AS cmt_status,
                prices.valid_until AS valid_until,
                prices.aoc AS aoc,
                prices.accessory_currency AS accessory_currency,
                format(prices.accessory, 2) AS accessory,
                prices.label_currency AS label_currency,
                format(prices.label, 2) AS label,
                prices.gmt_trans_currency AS gmt_trans_currency,
                format(prices.gmt_trans, 2) AS gmt_trans,
                prices.embroidery_currency AS embroidery_currency,
                format(prices.embroidery, 2) AS embroidery,
                format(prices.testing, 2) AS testing,
                format(prices.testing_courier, 2) AS testing_courier,
                prices.quote_status AS quote_status,
                format(prices.target, 2) AS target,
                prices.min_qty AS min_qty,
                prices.duty,
                prices.notes AS notes,
                prices.customer_notes AS customer_notes,
                suppliers.currency AS fact_currency,
                customers.currency AS cust_currency,
                customers.id AS customer_id,
                customers.customs AS customer_customs,
                suppliers.customs AS factory_customs,
                seasons.locked AS season_locked,
                (
                    SELECT
                        GROUP_CONCAT(
                            CONCAT(percentage, '% in ', days, ' days')
                            ORDER BY days ASC
                            SEPARATOR ', '
                        )
                    FROM
                        customer_payment_terms
                    WHERE
                        customer_payment_terms.customer_id = customers.id
                ) AS cust_terms,
                (
                    select
                        c.image
                    from
                            style_versions sv
                            left join colourways c on sv.id = style_versions.id
                    where c.style_versions_id = sv.id and (c.image is not null)
                    limit 1
                ) AS image,
                (
                    select
                        case when count(0) = 0
                        then 1
                        else count(0)
                        end
                    from
                        prices p
                    where p.style_versions_id = style_versions.id
                ) AS no_style_version_prices,
                COALESCE((
                    select
                        case when count(p.id) = 0
                        then 0
                        else 1
                        end
                    from
                        prices p
                    where p.style_versions_id = style_versions.id
                ), 0) AS got_prices,
                SUM(prices.id) OVER (PARTITION BY style_versions.id) as prices_count,
                (
                    select
                        group_concat(
                            distinct y.description separator ','
                        )
                    from yarns y
                    join yarn_colours yc on yc.yarn_id = y.id
                    join colourway_yarns cy on cy.yarn_colours_id = yc.id
                    join colourways c on c.id = cy.colourways_id
                    join style_versions sv on sv.id = c.style_versions_id
                    where sv.id = style_versions.id
                ) AS yarn,
                (
                    select
                        concat(
                            '[',
                            group_concat(
                                json_object(
                                    'name', c.name, 'id', c.id
                                ) separator ','
                            ),
                            ']'
                        )
                    from
                        colourways c
                    where c.style_versions_id = style_versions.id
                ) AS colourways,





                (
                    SELECT
                        CONCAT('[', GROUP_CONCAT(
                            JSON_OBJECT(
                                'season', prev_season.description,
                                'price_id', prev_p.id,
                                'cmt', prev_p.cmt,
                                'quote', prev_p.quote
                            )
                            ORDER BY prev_season.id DESC
                            SEPARATOR ','
                        ), ']')
                    FROM prices AS prev_p
                    JOIN style_versions AS prev_sv ON prev_p.style_versions_id = prev_sv.id
                    JOIN styles AS prev_s ON prev_sv.styles_id = prev_s.id
                    JOIN seasons AS prev_season ON prev_season.id=prev_s.seasons_id
                    WHERE prev_s.designs_id = styles.designs_id
                        AND prev_s.seasons_id < seasons.id
                        AND prev_p.sizes_id = prices.sizes_id
                        AND prev_p.quote_status = 'confirmed'
                    ORDER BY prev_season.id DESC
                    LIMIT 3
                ) AS historic,



                prices.quotes_sent AS quotes_sent,
                prices.weight_comments AS weight_comments,
                prices.cost_comments AS cost_comments,
                format(prices.quote, 2) AS quote,
                convert_currency(prices.quote, customers.currency, '£', styles.seasons_id) AS quote_gbp,
                suppliers.countries_id AS coo,
                countries.country AS coo_name,
                styles.departments_id AS departments_id,
                styles.category AS category,
                styles.cancelled AS style_cancelled,
                sizes.name AS sizes_name,
                colourways.name AS colourways_name,
                (
                    select users.name
                    from audits
                    join users on users.id = audits.user_id
                    where audits.auditable_type = 'App\\\Models\\\Price'
                    and audits.auditable_id = prices.id
                    order by audits.created_at desc
                    limit 1
                ) AS last_updated_by,






                0 AS yarn_cost_kg_eur -- will be computed in PHP using the colourways relationship


            from styles
            join seasons on seasons.id = styles.seasons_id
            join designs on designs.id = styles.designs_id
            join style_versions on style_versions.styles_id = styles.id
            join suppliers on suppliers.id = style_versions.factory_id
            join countries on countries.id = suppliers.countries_id
            join customers on customers.id = styles.customers_id
            left join intakes on intakes.id = styles.intake_id
            left join prices on prices.style_versions_id = style_versions.id and prices.deleted_at IS NULL
            left join phases on phases.id = prices.phase_id
            left join sizes on sizes.id = prices.sizes_id
            left join colourways on colourways.id = prices.colourways_id ";
            foreach($where as $wr){
                $query = $query . $wr . " \n";
            }
            $query = $query . "
            order by
            style_versions.styles_id,
            style_versions.id,
            sizes.order
        ";

        $this->style = collect(DB::select($query));

        // Add yarn_cost_kg_eur for each style by loading the colourway relationship
        $this->style = $this->style->map(function ($style) {
            if (isset($style->colourways_id) && $style->colourways_id) {
                $colourway = \App\Models\Colourways::with('colourway_yarns.yarn_colours.yarn.suppliers', 'style_versions.styles.seasons')
                    ->find($style->colourways_id);
                $style->yarn_cost_kg_eur = $colourway ? $colourway->yarn_cost_kg_eur : 0;
            } else {
                $style->yarn_cost_kg_eur = 0;
            }
            return $style;
        });

        $this->style = json_decode(json_encode($this->style->groupBy(['style_version_id'])),true);
    }
}
