-- Migration 008: Add Home Page Customization System
-- This migration adds support for customizable home page layouts

-- Table to store customization element types (text, image, social links, etc.)
CREATE TABLE IF NOT EXISTS customization_elements (
    id SERIAL PRIMARY KEY,
    element_key VARCHAR(100) UNIQUE NOT NULL,  -- e.g., 'hero_title', 'about_text', 'instagram_url'
    element_type VARCHAR(50) NOT NULL,          -- 'text', 'textarea', 'image', 'url', 'color', 'rich_text'
    label VARCHAR(255) NOT NULL,                -- Display name for admin interface
    description TEXT,                            -- Help text for users
    default_value TEXT,                          -- Default value if not customized
    validation_rules JSONB,                      -- JSON validation rules (e.g., max length, URL format)
    display_order INTEGER DEFAULT 0,
    is_active BOOLEAN DEFAULT true,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

-- Table to store layout templates created by superuser
CREATE TABLE IF NOT EXISTS layout_templates (
    id SERIAL PRIMARY KEY,
    layout_key VARCHAR(100) UNIQUE NOT NULL,    -- e.g., 'classic', 'modern', 'minimal'
    name VARCHAR(255) NOT NULL,                  -- Display name
    description TEXT,                            -- Description of the layout
    preview_image VARCHAR(500),                  -- URL to preview image
    template_file VARCHAR(255) NOT NULL,         -- EJS template file name
    is_active BOOLEAN DEFAULT true,
    is_default BOOLEAN DEFAULT false,
    display_order INTEGER DEFAULT 0,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

-- Junction table mapping which elements are used in which layouts
CREATE TABLE IF NOT EXISTS layout_elements (
    id SERIAL PRIMARY KEY,
    layout_id INTEGER NOT NULL REFERENCES layout_templates(id) ON DELETE CASCADE,
    element_id INTEGER NOT NULL REFERENCES customization_elements(id) ON DELETE CASCADE,
    is_required BOOLEAN DEFAULT false,
    section_name VARCHAR(100),                   -- Group elements by section (e.g., 'hero', 'about', 'contact')
    display_order INTEGER DEFAULT 0,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    UNIQUE(layout_id, element_id)
);

-- Table to store user's home page customizations
CREATE TABLE IF NOT EXISTS user_home_customizations (
    id SERIAL PRIMARY KEY,
    user_id INTEGER NOT NULL REFERENCES users(id) ON DELETE CASCADE,
    layout_id INTEGER REFERENCES layout_templates(id) ON DELETE SET NULL,
    element_id INTEGER NOT NULL REFERENCES customization_elements(id) ON DELETE CASCADE,
    element_value TEXT,                          -- The actual value set by the user
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    UNIQUE(user_id, element_id)
);

-- Indexes for performance
CREATE INDEX IF NOT EXISTS idx_customization_elements_active ON customization_elements(is_active);
CREATE INDEX IF NOT EXISTS idx_layout_templates_active ON layout_templates(is_active);
CREATE INDEX IF NOT EXISTS idx_layout_elements_layout ON layout_elements(layout_id);
CREATE INDEX IF NOT EXISTS idx_layout_elements_element ON layout_elements(element_id);
CREATE INDEX IF NOT EXISTS idx_user_home_customizations_user ON user_home_customizations(user_id);
CREATE INDEX IF NOT EXISTS idx_user_home_customizations_layout ON user_home_customizations(layout_id);

-- Insert default customization elements
INSERT INTO customization_elements (element_key, element_type, label, description, default_value, display_order) VALUES
    ('hero_title', 'text', 'Hero Title', 'Main heading on your home page', 'Welcome to My Gallery', 1),
    ('hero_subtitle', 'text', 'Hero Subtitle', 'Subtitle or tagline', 'Professional Photography', 2),
    ('hero_image', 'image', 'Hero Background Image', 'Background image for hero section', NULL, 3),
    ('about_title', 'text', 'About Section Title', 'Title for about section', 'About Me', 4),
    ('about_text', 'textarea', 'About Text', 'Tell visitors about yourself and your work', 'I''m a professional photographer...', 5),
    ('about_image', 'image', 'About Image', 'Image for about section', NULL, 6),
    ('contact_title', 'text', 'Contact Section Title', 'Title for contact section', 'Get In Touch', 7),
    ('contact_text', 'textarea', 'Contact Text', 'Contact information or message', 'I''d love to hear from you!', 8),
    ('instagram_url', 'url', 'Instagram URL', 'Your Instagram profile URL', NULL, 9),
    ('facebook_url', 'url', 'Facebook URL', 'Your Facebook page URL', NULL, 10),
    ('twitter_url', 'url', 'Twitter/X URL', 'Your Twitter/X profile URL', NULL, 11),
    ('pinterest_url', 'url', 'Pinterest URL', 'Your Pinterest profile URL', NULL, 12),
    ('email_address', 'text', 'Contact Email', 'Your contact email address', NULL, 13),
    ('phone_number', 'text', 'Phone Number', 'Your contact phone number', NULL, 14),
    ('cta_text', 'text', 'Call-to-Action Button Text', 'Text for main CTA button', 'View Gallery', 15),
    ('cta_link', 'url', 'Call-to-Action Button Link', 'Link for main CTA button', '/albums', 16),
    ('background_color', 'color', 'Background Color', 'Background color for sections', '#ffffff', 17),
    ('accent_color', 'color', 'Accent Color', 'Accent color for buttons and highlights', '#000000', 18),
    ('custom_css', 'textarea', 'Custom CSS', 'Advanced: Add custom CSS styles', NULL, 19),
    ('show_recent_albums', 'checkbox', 'Show Recent Albums', 'Display recent albums on home page', 'true', 20)
ON CONFLICT (element_key) DO NOTHING;

-- Insert default layout templates
INSERT INTO layout_templates (layout_key, name, description, template_file, is_default, display_order) VALUES
    ('classic', 'Classic', 'Traditional layout with hero section, about, and contact', 'home-classic.ejs', true, 1),
    ('modern', 'Modern', 'Contemporary design with bold typography and minimal elements', 'home-modern.ejs', false, 2),
    ('minimal', 'Minimal', 'Clean and simple layout focusing on your work', 'home-minimal.ejs', false, 3),
    ('full_width', 'Full Width', 'Full-width hero with immersive experience', 'home-fullwidth.ejs', false, 4)
ON CONFLICT (layout_key) DO NOTHING;

-- Map elements to Classic layout
INSERT INTO layout_elements (layout_id, element_id, is_required, section_name, display_order)
SELECT 
    (SELECT id FROM layout_templates WHERE layout_key = 'classic'),
    ce.id,
    CASE 
        WHEN ce.element_key IN ('hero_title', 'hero_subtitle') THEN true
        ELSE false
    END,
    CASE 
        WHEN ce.element_key LIKE 'hero_%' THEN 'hero'
        WHEN ce.element_key LIKE 'about_%' THEN 'about'
        WHEN ce.element_key LIKE 'contact_%' THEN 'contact'
        WHEN ce.element_key LIKE '%_url' OR ce.element_key IN ('email_address', 'phone_number') THEN 'social'
        WHEN ce.element_key LIKE 'cta_%' THEN 'cta'
        ELSE 'general'
    END,
    ce.display_order
FROM customization_elements ce
WHERE ce.element_key IN (
    'hero_title', 'hero_subtitle', 'hero_image',
    'about_title', 'about_text', 'about_image',
    'contact_title', 'contact_text',
    'instagram_url', 'facebook_url', 'twitter_url',
    'email_address', 'cta_text', 'cta_link',
    'show_recent_albums'
)
ON CONFLICT DO NOTHING;

-- Map elements to Modern layout
INSERT INTO layout_elements (layout_id, element_id, is_required, section_name, display_order)
SELECT 
    (SELECT id FROM layout_templates WHERE layout_key = 'modern'),
    ce.id,
    CASE 
        WHEN ce.element_key IN ('hero_title') THEN true
        ELSE false
    END,
    CASE 
        WHEN ce.element_key LIKE 'hero_%' THEN 'hero'
        WHEN ce.element_key LIKE 'about_%' THEN 'about'
        WHEN ce.element_key LIKE '%_url' THEN 'social'
        WHEN ce.element_key LIKE '%_color' THEN 'style'
        ELSE 'general'
    END,
    ce.display_order
FROM customization_elements ce
WHERE ce.element_key IN (
    'hero_title', 'hero_subtitle',
    'about_text',
    'instagram_url', 'facebook_url', 'twitter_url', 'pinterest_url',
    'cta_text', 'cta_link',
    'background_color', 'accent_color',
    'show_recent_albums'
)
ON CONFLICT DO NOTHING;

-- Map elements to Minimal layout
INSERT INTO layout_elements (layout_id, element_id, is_required, section_name, display_order)
SELECT 
    (SELECT id FROM layout_templates WHERE layout_key = 'minimal'),
    ce.id,
    CASE 
        WHEN ce.element_key IN ('hero_title') THEN true
        ELSE false
    END,
    CASE 
        WHEN ce.element_key LIKE 'hero_%' THEN 'hero'
        WHEN ce.element_key LIKE '%_url' THEN 'social'
        ELSE 'general'
    END,
    ce.display_order
FROM customization_elements ce
WHERE ce.element_key IN (
    'hero_title', 'hero_subtitle',
    'instagram_url', 'email_address',
    'cta_text', 'cta_link',
    'show_recent_albums'
)
ON CONFLICT DO NOTHING;

-- Map elements to Full Width layout
INSERT INTO layout_elements (layout_id, element_id, is_required, section_name, display_order)
SELECT 
    (SELECT id FROM layout_templates WHERE layout_key = 'full_width'),
    ce.id,
    CASE 
        WHEN ce.element_key IN ('hero_title', 'hero_image') THEN true
        ELSE false
    END,
    CASE 
        WHEN ce.element_key LIKE 'hero_%' THEN 'hero'
        WHEN ce.element_key LIKE '%_url' THEN 'social'
        WHEN ce.element_key LIKE 'cta_%' THEN 'cta'
        ELSE 'general'
    END,
    ce.display_order
FROM customization_elements ce
WHERE ce.element_key IN (
    'hero_title', 'hero_subtitle', 'hero_image',
    'instagram_url', 'facebook_url', 'twitter_url', 'pinterest_url',
    'cta_text', 'cta_link',
    'show_recent_albums'
)
ON CONFLICT DO NOTHING;

-- Add a column to users table to store their selected layout (if not exists)
DO $$ 
BEGIN
    IF NOT EXISTS (SELECT 1 FROM information_schema.columns 
                   WHERE table_name='users' AND column_name='home_layout_id') THEN
        ALTER TABLE users ADD COLUMN home_layout_id INTEGER REFERENCES layout_templates(id) ON DELETE SET NULL;
    END IF;
END $$;

COMMENT ON TABLE customization_elements IS 'Defines available customization elements that can be used in layouts';
COMMENT ON TABLE layout_templates IS 'Stores layout templates created by superuser';
COMMENT ON TABLE layout_elements IS 'Maps which customization elements are used in each layout';
COMMENT ON TABLE user_home_customizations IS 'Stores user-specific values for customization elements';

