<?php

namespace App\Models;

use App\Casts\TitleCase;
use App\Helper\Functions;
use \Bkwld\Cloner\Cloneable;
use Illuminate\Support\Facades\Gate;
use App\Models\BaseModel;
use OwenIt\Auditing\Contracts\Auditable;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use App\Services\TotalCacheService;

class CustomerOrderLines extends BaseModel implements Auditable
{
    use \OwenIt\Auditing\Auditable;
    use SoftDeletes;
    use Cloneable;

    public static function boot()
    {
        parent::boot();

        static::deleting(function($model) {
            $model->shipment_lines?->each?->delete();
            $model->customer_order_line_quantities?->each?->delete();
            $model->yarn_order_line_styles?->each?->delete();
        });
        static::restoring(function($model) {
            $model->shipment_lines()->withTrashed()->where('deleted_at', '>=', $model->deleted_at)->each(function ($item, $key) { $item->restore(); });
            $model->customer_order_line_quantities()->withTrashed()->where('deleted_at', '>=', $model->deleted_at)->each(function ($item, $key) { $item->restore(); });
            $model->yarn_order_line_styles()->withTrashed()->where('deleted_at', '>=', $model->deleted_at)->each(function ($item, $key) { $item->restore(); });
        });

        // Total cache invalidation hooks
        static::saved(function($model) {
            $model->invalidateTotalCache();
        });

        static::updated(function($model) {
            $model->invalidateTotalCache();
        });

        static::deleted(function($model) {
            $model->invalidateTotalCache();
        });
    }

    protected $appends = ['total_value'];

    protected $casts = [
    	'factory_cust_date' => 'date:Y-m-d',
    	'start_knit' => 'date:Y-m-d',
    	'actual_ship_date' => 'date:Y-m-d',
    	'wh_cust_date' => 'date:Y-m-d',
    	'colour_approved_date' => 'date:Y-m-d',
    	'sent_to_factory' => 'date:Y-m-d',
    	'lab_dips_sent_date' => 'date:Y-m-d',
        'location' => TitleCase::class,
        'created_at' => 'datetime:Y-m-d',
        'updated_at' => 'datetime:Y-m-d',
        'deleted_at' => 'datetime:Y-m-d',
    ];

    protected $fillable = [
        'customer_orders_id',
        'colourways_id',
        'location',
        'notes',
        'actual_ship_date',
        'sent_to_factory',
        // 'last_updated_by',
        'created_at',
        'updated_at',
        'factory_cust_date',
        'cp_notes',
        'colour_approved_date',
        'lab_dips_sent_date',
        'colour_approval_comments',
        'wh_cust_date',
        'start_knit',
        'deleted_at',
        'bulk_yarn_order_date',
        'bulk_yarn_order_comments',
        'bulk_yarn_order_delivery_date',
        'bulk_yarn_order_delivery_comments',
        'order_sent_factory',
        'cancelled',
    ];

    // public function saveWithUser(){
    //     $this->last_updated_by = auth()->user()->id;
    //     $this->save();
    // }

    public function totalQty(){
		return $this->customer_order_line_quantities->sum('qty');
	}
    public function totalQtyDrops(){
        $dropsQty = 0;
        foreach($this->shipment_lines as $line){
            $dropsQty = $dropsQty + $line->shipment_line_sizes->sum('qty');
        }
        return $dropsQty;
	}
    // public function dropsMatchOrder(){
    //     if($this->totalQty() != $this->totalQtyDrops())
    //         return FALSE;
    //     else
    //         return TRUE;
    // }

    public function getTotalValueAttribute(){
        // Use total cache system instead of manual calculation
        $service = app(TotalCacheService::class);
        $totals = $service->get('customer_order_line', $this->id, 'prices');
        
        if ($totals) {
            return $totals['quote'] ?? 0;
        }
        
        // Fallback: warm up cache and return the value
        $service->warmupCustomerOrderLine($this);
        $totals = $service->get('customer_order_line', $this->id, 'prices');
        
        return $totals['quote'] ?? 0;
    }

    // public function getTotalPiecesAttribute(){
    //     dd($this->customer_order_line_quantities?->sum('qty'));
    //     return $this->customer_order_line_quantities?->sum('qty');
    // }

    public function getTotalPiecesAttribute()
    {
        // Check if the relationship is loaded to avoid lazy-loading and potential hangs
        if ($this->relationLoaded('customer_order_line_quantities')) {
            // Use collection sum if already loaded
            return $this->customer_order_line_quantities->sum('qty');
        } else {
            // If not loaded, fetch the sum directly from the database to avoid loading all records
            return $this->customer_order_line_quantities()->sum('qty');
        }
    }


    // public function latestCustomerDate(){
    //     if($this->customer_order_line_cust_dates->sortBy('created_at')->last() != NULL)
    //         return $this->customer_order_line_cust_dates->sortBy('created_at')->last()->date;
    // }

    // public function latestStartKnitDate(){
    //     if($this->customer_order_line_knit_dates->sortBy('created_at')->last() != NULL)
    //         return $this->customer_order_line_knit_dates->sortBy('created_at')->last()->date;
    // }

	// public function setBrandSizeLabelsDoneAttribute($value)
	// {
	// 	$this->attributes['brand_size_labels_done'] = $value ?? FALSE;
	// }
	// public function setCareLabelsDoneAttribute($value)
	// {
	// 	$this->attributes['care_labels_done'] = $value ?? FALSE;
	// }
	// public function setBarcodesDoneAttribute($value)
	// {
	// 	$this->attributes['barcodes_done'] = $value ?? FALSE;
	// }
	// public function setAccessoriesDoneAttribute($value)
	// {
	// 	$this->attributes['accessories_done'] = $value ?? FALSE;
	// }

    public function customer_orders()
    {
        return $this->belongsTo(CustomerOrders::class);
    }
    public function colourways()
    {
        return $this->belongsTo(Colourways::class, 'colourways_id');
    }

	public function customer_order_line_quantities()
    {
        return $this->hasMany(CustomerOrderLineQuantities::class);
    }
    public function shipment_lines()
    {
        return $this->hasMany(ShipmentLine::class);
    }
    public function yarn_order_line_styles(){
        return $this->hasMany(YarnOrderLineStyles::class);
    }

    /**
     * Invalidate total cache for this order line and related entities.
     */
    public function invalidateTotalCache(): void
    {
        try {
            $service = app(TotalCacheService::class);
            
            // Invalidate this customer order line's cache
            $service->invalidateEntity('customer_order_line', $this->id);
            
            // Invalidate related customer order cache
            if ($this->customer_orders_id) {
                $service->invalidateEntity('customer_order', $this->customer_orders_id);
            }
            
            // Invalidate related shipment line caches
            $shipmentLineIds = $this->shipment_lines()->pluck('id')->toArray();
            if (!empty($shipmentLineIds)) {
                $service->invalidateEntities('shipment_line', $shipmentLineIds);
                
                // Also invalidate related shipment caches
                $shipmentIds = $this->shipment_lines()->pluck('shipment_id')->filter()->unique()->toArray();
                if (!empty($shipmentIds)) {
                    $service->invalidateEntities('shipment', $shipmentIds);
                }
            }
        } catch (\Exception $e) {
            \Log::error("Failed to invalidate total cache for customer order line {$this->id}: " . $e->getMessage());
        }
    }
}
