<?php

namespace Tests\Feature;

use App\Models\User;
use App\Models\Seasons;
use App\Models\Styles;
use App\Models\Sizes;
use App\Models\StyleVersions;
use App\Models\Price;
use App\Http\Livewire\Imports\PriceImport;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Http\UploadedFile;
use Illuminate\Support\Facades\Storage;
use Livewire\Livewire;
use Tests\TestCase;

class PriceImportTest extends TestCase
{
    use RefreshDatabase;

    public function test_component_can_be_rendered()
    {
        $user = User::factory()->create();
        $this->actingAs($user);

        $component = Livewire::test(PriceImport::class);
        $component->assertStatus(200);
    }

    public function test_season_selection_loads_styles()
    {
        $user = User::factory()->create();
        $this->actingAs($user);

        // Create test data
        $season = Seasons::factory()->create(['description' => 'AW25']);
        $style = Styles::factory()->create([
            'seasons_id' => $season->id,
            'customer_ref' => 'MAB733'
        ]);

        $component = Livewire::test(PriceImport::class)
            ->set('selectedSeason', $season->id);

        $this->assertCount(1, $component->styles);
        $this->assertEquals('MAB733', $component->styles->first()->customer_ref);
    }

    public function test_csv_file_processing()
    {
        $user = User::factory()->create();
        $this->actingAs($user);

        Storage::fake('local');

        // Create test data
        $season = Seasons::factory()->create(['description' => 'AW25']);
        $style = Styles::factory()->create([
            'seasons_id' => $season->id,
            'customer_ref' => 'MAB733'
        ]);
        StyleVersions::factory()->create(['styles_id' => $style->id]);

        $size = Sizes::factory()->create(['name' => '6M']);

        // Create test CSV content
        $csvContent = "Season\tStyle\tSize\tcmt\tquote\nAW25\tMAB733\t6M\t14.00\t14.90\n";
        
        $file = UploadedFile::fake()->createWithContent('test.csv', $csvContent);

        $component = Livewire::test(PriceImport::class)
            ->set('selectedSeason', $season->id)
            ->set('csvFile', $file)
            ->call('processCsvFile');

        $this->assertCount(1, $component->processedData);
        $this->assertEquals('MAB733', $component->processedData[0]['style']);
        $this->assertEquals('6M', $component->processedData[0]['size']);
        $this->assertEquals('14.00', $component->processedData[0]['cmt']);
        $this->assertEquals('14.90', $component->processedData[0]['quote']);
    }

    public function test_style_matching()
    {
        $user = User::factory()->create();
        $this->actingAs($user);

        // Create test data
        $season = Seasons::factory()->create(['description' => 'AW25']);
        $style = Styles::factory()->create([
            'seasons_id' => $season->id,
            'customer_ref' => 'MAB733'
        ]);
        StyleVersions::factory()->create(['styles_id' => $style->id]);

        $csvContent = "Season\tStyle\tSize\tcmt\tquote\nAW25\tMAB733\t6M\t14.00\t14.90\n";
        $file = UploadedFile::fake()->createWithContent('test.csv', $csvContent);

        $component = Livewire::test(PriceImport::class)
            ->set('selectedSeason', $season->id)
            ->set('csvFile', $file)
            ->call('processCsvFile');

        $this->assertTrue($component->processedData[0]['has_match']);
        $this->assertEquals($style->id, $component->processedData[0]['style_id']);
    }

    public function test_price_saving()
    {
        $user = User::factory()->create();
        $this->actingAs($user);

        // Create test data
        $season = Seasons::factory()->create(['description' => 'AW25']);
        $style = Styles::factory()->create([
            'seasons_id' => $season->id,
            'customer_ref' => 'MAB733'
        ]);
        $styleVersion = StyleVersions::factory()->create(['styles_id' => $style->id]);
        $size = Sizes::factory()->create(['name' => '6M']);

        $csvContent = "Season\tStyle\tSize\tcmt\tquote\nAW25\tMAB733\t6M\t14.00\t14.90\n";
        $file = UploadedFile::fake()->createWithContent('test.csv', $csvContent);

        $component = Livewire::test(PriceImport::class)
            ->set('selectedSeason', $season->id)
            ->set('csvFile', $file)
            ->call('processCsvFile')
            ->call('savePrices');

        $this->assertDatabaseHas('prices', [
            'style_versions_id' => $styleVersion->id,
            'cmt' => 14.00,
            'quote' => 14.90,
            'sizes_id' => $size->id,
            'model' => 'manual'
        ]);

        $component->assertHasNoErrors();
    }
}
